// 编译命令
// gcc -o ten264_example_region ten264_example_region.c -I ../lib/ -L ../lib/ -ltscsdk_center -ldl -lpthread -lm -lz -lrt
// 运行方法
// ten264_example_region nv12.yuv ten264_example_region [rate_control_method: crf or vbv_crf]
// 例如
// ten264_example_region nv12.yuv ten264_example_region.h264 crf
// ten264_example_region nv12.yuv ten264_example_region.h264 vbv_crf

#include "ten264_example_common.h"


/*************************** different code ***************************/
void roi_mask_clear(float *quant_offsets, int pic_width, int pic_height) {
    int quant_offsets_size = ((pic_width + 15) >> 4) * ((pic_height + 15) >> 4);
    memset(quant_offsets, 0, quant_offsets_size * sizeof(float));
}
float *roi_mask_alloc(int pic_width, int pic_height) {
    int quant_offsets_size = ((pic_width + 15) >> 4) * ((pic_height + 15) >> 4);
    float *quant_offsets = (float *)malloc(quant_offsets_size * sizeof(float));
    roi_mask_clear(quant_offsets, pic_width, pic_height);
    return quant_offsets;
}
void roi_mask_set_one_box(float *quant_offsets, int pic_width, int pic_height,
        int x_top_left, int y_top_left, int x_bottom_right, int y_bottom_right) {
    int mb_width = (pic_width + 15) >> 4;
    int mb_height = (pic_height + 15) >> 4;
    // use offset to affect inner microblocks only
    int coor_offset = 8;
    int mbx_top_left = (x_top_left + coor_offset) / 16;
    int mby_top_left = (y_top_left + coor_offset) / 16;
    int mbx_bottom_right = (x_bottom_right - coor_offset) / 16;
    int mby_bottom_right = (y_bottom_right - coor_offset) / 16;
    int col, row;
    for (col = mbx_top_left; col <= mbx_bottom_right; col++) {
        for (row = mby_top_left; row <= mby_bottom_right; row++) {
            quant_offsets[row * mb_width + col] = 1;
        }
    }
}
/*************************** different code ***************************/


int main(int argc, char **argv) {
    FILE *file_in = NULL, *file_out = NULL;
    int width = 1920, height = 1080;
    char *rc_method = "crf";
    ten264_param_t params;
    ten264_t *enc = NULL;
    void *extra = NULL;
    ten264_picture_t pic;

    /* Check args */
    FAIL_IF_ERROR(parse_args(argc, argv, &rc_method) != 0, "Example usage: \n"
        "ten264_example input.yuv output.h264 "
        "[rate_control_method: crf or vbv_crf] \n");

    /* Open files */
    FAIL_IF_ERROR(open_file(&file_in, argv[1], &file_out, argv[2]) != 0, "");

    /* Configure base params */
    FAIL_IF_ERROR(ten264example_prarm_init(&params, width, height) != 0, "");

    /* crf参数可选： 21.5, 26.6, 31.7, 36 */
    if (strcmp(rc_method, "vbv_crf") == 0) {
        FAIL_IF_ERROR(ten264example_rc_method_vbv_crf(&params, "21.5", 400) != 0, "");
    } else if (strcmp(rc_method, "crf") == 0) {
        FAIL_IF_ERROR(ten264example_rc_method_crf(&params, "21.5") != 0, "");
    }

    /* Parse params */
    const char ten264_param[] = "threads=4:keyint=infinite:p-opt=1,0,4,1.5:tune=zerolatency_shxf";
    FAIL_IF_ERROR(ten264example_param_parse(&params, ten264_param) != 0, "");
    
/*************************** different code ***************************/
    /* Open roi function */
    FAIL_IF_ERROR(ten264_param_parse(&params, "roi", "0.5") < 0, 
        "ten264_param_parse failed. \n");
/*************************** different code ***************************/

    /* Open encoder */
    FAIL_IF_ERROR(!(enc = ten264_encoder_open(&params, &extra, "../sdk_config")),
        "ten264_encoder_open failed.\n");

    /* Configure input picture */
    ten264example_picture_init_nv12(&pic, &params);

/*************************** different code ***************************/
    /* Alloc quant_offsets for roi function */
    pic.prop.quant_offsets = roi_mask_alloc(width, height);
/*************************** different code ***************************/

    /* Get h264 headers */
    FAIL_IF_ERROR(ten264example_get_header(enc, file_out) != 0, "");
    
    /* Encode frames */
    while (true) {

/*************************** different code ***************************/
        roi_mask_clear(pic.prop.quant_offsets, width, height);
        /* set a region in frame center */
        roi_mask_set_one_box(pic.prop.quant_offsets, width, height,
            width / 2 - 200, height / 2 - 200, width / 2 + 200, height / 2 + 200);
/*************************** different code ***************************/

        /* Read input frame */
        if (ten264example_read_frame_nv12(&pic, &params, file_in) != 0)
            break;

        /* Update pts */
        pic.i_pts++;

        /* Encode frame */
        FAIL_IF_ERROR(ten264example_encode_frame(enc, &pic, extra, file_out) != 0, "");
    }

    /* Flush delayed frames */
    FAIL_IF_ERROR(ten264example_flush_encoder(enc, extra, file_out) != 0, "");

    /* Exit program */
    int ret = 0;
    goto success;

fail:
    ret = -1;
success:
    if (enc)
        ten264_encoder_close(enc, &extra);
/*************************** different code ***************************/
    if (pic.prop.quant_offsets)
        free(pic.prop.quant_offsets);
/*************************** different code ***************************/
    ten264example_picture_free(&pic);
    close_file(&file_in, &file_out);
    return ret;
}
