#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <libtscsdk.h>


/*************************** common functions ***************************/


#define RET_IF_ERROR(ret, cond, ...)        \
do {                                        \
    if ((cond)) {                           \
        fprintf(stderr, __VA_ARGS__);       \
        return ret;                         \
    }                                       \
} while (0)


#define GOTO_IF_ERROR(position, cond, ...)  \
do {                                        \
    if ((cond)) {                           \
        fprintf(stderr, __VA_ARGS__);       \
        goto position;                      \
    }                                       \
} while (0)


#define FAIL_IF_ERROR(cond, ...) GOTO_IF_ERROR(fail, cond, __VA_ARGS__)


void close_file(FILE **file_in, FILE **file_out) {
    if (*file_in) {
        fclose(*file_in);
        *file_in = NULL;
    }
    if (*file_out) {
        fclose(*file_out);
        *file_out = NULL;
    }
}


int open_file(FILE **file_in, const char *path_in,
        FILE **file_out, const char *path_out) {
    GOTO_IF_ERROR(open_file_fail, (*file_in = fopen(path_in, "rb")) == NULL, "open input file failed.\n");
    GOTO_IF_ERROR(open_file_fail, (*file_out = fopen(path_out, "wb")) == NULL, "open output file failed.\n");
    return 0;

open_file_fail:
    close_file(file_in, file_out);
    return -1;
}


int parse_args(int argc, char **argv, char **rc_method) {
    if (argc < 3)
        return -1;
    int i;
    for (i = 3; i < argc; i++) {
        if (strcmp(argv[i], "crf") == 0 || strcmp(argv[i], "vbv_crf") == 0)
            *rc_method = argv[i];
    }
    return 0;
}


/*************************** ten264 common functions ***************************/


void ten264example_log_default(void *p_unused, int i_level, const char *psz_fmt, va_list arg) {
    static const char *psz_prefix[] = {
        [TEN264_LOG_NONE+1]    = "none",
        [TEN264_LOG_ERROR+1]   = "error",
        [TEN264_LOG_WARNING+1] = "warning",
        [TEN264_LOG_INFO+1]    = "info",
        [TEN264_LOG_DEBUG+1]   = "debug"
    };
    const char *level = ((i_level+1) >= 0 && (i_level+1) <= (TEN264_LOG_DEBUG+1))
        ? psz_prefix[i_level+1] : "none";
    fprintf(stderr, "ten264 [%s]: ", level);
    vfprintf(stderr, psz_fmt, arg);
}


int ten264example_prarm_init(ten264_param_t *params, int width, int height) {
    RET_IF_ERROR(-1, ten264_param_default_preset(params, "ultrafast") < 0,
        "ten264_param_default_preset failed.\n");
    params->i_width  = width;
    params->i_height = height;
    params->i_bitdepth = 8;
    params->i_csp = TEN264_CSP_NV12;
    params->b_vfr_input = 0;
    params->b_repeat_headers = 0;

    params->pf_log = ten264example_log_default;
    params->i_log_level = TEN264_LOG_DEBUG;

    params->i_timebase_den = 24;
    params->i_timebase_num = 1;
    params->i_fps_num = 24;
    params->i_fps_den = 1;
    return 0;
}


int ten264example_rc_method_crf(ten264_param_t *params, const char *crf_val) {
    params->rc.i_rc_method = TEN264_RC_CRF;
    RET_IF_ERROR(-1, ten264_param_parse(params, "crf", crf_val) < 0, 
            "ten264_param_parse crf failed.\n");
    return 0;
}


int ten264example_rc_method_vbv_crf(ten264_param_t *params, const char *crf_val, int vbv_val) {
    RET_IF_ERROR(-1, ten264example_rc_method_crf(params, crf_val) < 0, "");
    params->rc.i_vbv_max_bitrate = vbv_val;
    params->rc.i_vbv_buffer_size = vbv_val;
    return 0;
}


int ten264example_param_parse(ten264_param_t *params, const char *params_str) {
    int params_len = strlen(params_str);
    char key[128] = {0}, val[128] = {0};
    int offset = 0;
    while (offset < params_len) {
        if (sscanf(params_str + offset, "%127[^:=]=%127[^:]", key, val) == 2) {
            RET_IF_ERROR(-1, ten264_param_parse(params, key, val) < 0, 
                "ten264_param_parse failed. key=%s, val=%s\n", key, val);
        }
        while (params_str[offset] != ':' && offset < params_len)
            offset++;
        offset++;
    }
    return 0;
}


void ten264example_picture_init_nv12(ten264_picture_t *pic, ten264_param_t *params) {
    int luma_size = params->i_width * params->i_height;     // Y
    int chroma_size = luma_size / 2;    // UV
    ten264_picture_init(pic);
    pic->img.i_plane = 2;
    pic->img.plane[0] = malloc(luma_size);   // Y
    pic->img.plane[1] = malloc(chroma_size); // UV
    pic->img.i_stride[0] = params->i_width;
    pic->img.i_stride[1] = params->i_width;
    pic->img.i_csp = params->i_csp;
    pic->i_pts = 0;
    pic->prop.quant_offsets = NULL;
}


void ten264example_picture_free(ten264_picture_t *pic) {
    int i;
    for (i = 0; i < 4; i++)
        if (pic->img.plane[i])
            free(pic->img.plane[i]);
}


int ten264example_read_frame_nv12(ten264_picture_t *pic, ten264_param_t *params, FILE *file_in) {
    int luma_size = params->i_width * params->i_height;     // Y
    int chroma_size = luma_size / 2;    // UV
    RET_IF_ERROR(-1, fread(pic->img.plane[0], 1, luma_size, file_in) != (unsigned)luma_size, 
        "ten264example_read_frame EOF.\n");
    RET_IF_ERROR(-1, fread(pic->img.plane[1], 1, chroma_size , file_in) != (unsigned)chroma_size, 
        "ten264example_read_frame EOF.\n");
    return 0;
}


int ten264example_get_header(ten264_t *enc, FILE *file_out) {
    ten264_nal_t *nal;
    int i_nal;
    int i_frame_size = 0;
    i_frame_size = ten264_encoder_headers(enc, &nal, &i_nal);
    if (i_frame_size) {
        RET_IF_ERROR(-1, !fwrite(nal->p_payload, i_frame_size, 1, file_out), "write output file fail.\n");
    }
    return 0;
}


int ten264example_encode_frame(ten264_t *enc, ten264_picture_t *pic, void *extra, FILE *file_out) {
    ten264_picture_t pic_out;
    ten264_nal_t *nal;
    int i_nal;
    int i_frame_size = 0;
    i_frame_size = ten264_encoder_encode(enc, &nal, &i_nal, pic, &pic_out, NULL, 0, 0, &extra);
    RET_IF_ERROR(-1, i_frame_size < 0, "ten264_encoder_encode fail.\n");
    if (i_frame_size > 0) {
        RET_IF_ERROR(-1, !fwrite(nal->p_payload, i_frame_size, 1, file_out), "write output file fail.\n");
    }
    return 0;
}


int ten264example_flush_encoder(ten264_t *enc, void *extra, FILE *file_out) {
    while (ten264_encoder_delayed_frames(enc)) {
        RET_IF_ERROR(-1, ten264example_encode_frame(enc, NULL, extra, file_out), "");
    }
    return 0;
}

