// 编译命令
// gcc -o ten265_example ten265_example.c -I ../lib/ -L ../lib/ -ltscsdk_center -ldl -lpthread -lm -lz
// 运行方法
// ten265_example input.yuv output.h265 [rate_control_method: crf or vbv_crf] [extra_func: subjective or reconfig]
// 例如
// ten265_example input.yuv output.h265 crf

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <libtscsdk.h>

#define FAIL_IF_ERROR(cond, ...)\
do {\
    if ((cond)) {\
        fprintf(stderr, __VA_ARGS__);\
        goto fail;\
    }\
} while (0)

void ten265_log_default(void *p_unused, int i_level, const char *psz_fmt, va_list arg) {
    static const char *psz_prefix[] = {
        [TEN265_LOG_NONE+1]    = "none",
        [TEN265_LOG_ERROR+1]   = "error",
        [TEN265_LOG_WARNING+1] = "warning",
        [TEN265_LOG_INFO+1]    = "info",
        [TEN265_LOG_DEBUG+1]   = "debug"
    };
    fprintf(stderr, "ten265 [%s]: ", psz_prefix[i_level+1]);
    vfprintf(stderr, psz_fmt, arg);
}

enum TI265_RC_METHODS
{
    TEN265_RC_CQP = 0,
    TEN265_RC_ABR_VBV = 1,
    TEN265_RC_ABR = 2,
    TEN265_RC_CRF = 3
};


int main(int argc, char **argv) {
    FAIL_IF_ERROR(argc < 3, "Example usage: \n"
        "ten265_example input.yuv output.h265 "
        "[rate_control_method: crf or vbv_crf] "
        "[extra_func: subjective or reconfig] \n");
    char *extra_func = "";
    char *rc_method = "crf";
    int i;
    for (i = 3; i < argc; i++) {
        if (strcmp(argv[i], "subjective") == 0 || strcmp(argv[i], "reconfig") == 0)
            extra_func = argv[i];
        if (strcmp(argv[i], "crf") == 0 || strcmp(argv[i], "vbv_crf") == 0)
            rc_method = argv[i];
    }

    ten265_param_t *params;
    ten265_inpic_t pic;
    ten265_outpic_t *pic_out;
    ten265_t *enc;
    int i_frame = 0;
    int i_frame_size;
    ten265_nal_t *nal;
    int i_nal;
    void *extra;
    FILE *file_in, *file_out;
    int ret;
    int width = 1920, height = 1080;

    FAIL_IF_ERROR((file_in = fopen(argv[1], "rb")) == NULL, "open input file failed.\n");
    FAIL_IF_ERROR((file_out = fopen(argv[2], "wb")) == NULL, "open output file failed.\n");

    /* Configure params */
    FAIL_IF_ERROR(!(params = ten265_param_alloc()) < 0,
        "ten265_param_alloc failed.\n");
    ten265_param_default(params);

    params->m_iSourceWidth  = width;
    params->m_iSourceHeight = height;
    params->m_bRepeatHeaders = 1;
    params->pf_log = ten265_log_default;
    params->m_logLevel = TEN265_LOG_DEBUG;

    params->m_iTimeBaseDen = 24;
    params->m_iTimeBaseNum = 1;
    params->m_iFpsNum = 24;
    params->m_iFpsDenom = 1;

    // crf参数可选： 26, 30, 36, 40
    if (strcmp(rc_method, "vbv_crf") == 0) {
        params->m_iRateControlMode = TEN265_RC_CRF;
        FAIL_IF_ERROR(ten265_param_parse(params, "crf", "26") < 0, 
                "ten265_param_parse crf failed.\n");
        params->m_vbvMaxBitrate = 400;
        params->m_vbvBufferSize = 400;
    } else if (strcmp(rc_method, "crf") == 0) {
        params->m_iRateControlMode = TEN265_RC_CRF;
        FAIL_IF_ERROR(ten265_param_parse(params, "crf", "26") < 0, 
                "ten265_param_parse crf failed.\n");
    }

    const char ten265_param[] = "tune=shortlatency:preset=9:keyint=1000:ltr=0:max-dpb=16:aq-mode=1:level=5:gop-struct=lowdelayB:pool-threads=4";
    int ten265_param_len = strlen(ten265_param);
    char key[128] = {0};
    char val[128] = {0};
    int offset = 0;
    while (offset < ten265_param_len) {
        if (sscanf(ten265_param + offset, "%127[^:=]=%127[^:]", key, val) == 2) {
            FAIL_IF_ERROR(ten265_param_parse(params, key, val) < 0, 
                "ten265_param_parse failed. key=%s, val=%s\n", key, val);
        }
        while (ten265_param[offset] != ':' && offset < ten265_param_len)
            offset ++;
        offset++;
    }

    FAIL_IF_ERROR(!(enc = ten265_encoder_open(params, &extra, 0, "../sdk_config")),
        "ten265_encoder_open failed.\n");

    if (strcmp(extra_func, "subjective") == 0) {
        FAIL_IF_ERROR(add_preprocess(&extra, "sharp_size=3:sharp_amount=0.5") != 0,
            "add_preprocess failed.\n");
    }

    /* Configure input picture */
    int luma_size = width * height;     // Y
    int chroma_size = luma_size / 4;    // UV

    pic.plane[0] = malloc(luma_size);   // Y
    pic.plane[1] = malloc(chroma_size); // U
    pic.plane[2] = malloc(chroma_size); // V
    pic.stride[0] = width;
    pic.stride[1] = width / 2;
    pic.stride[2] = width / 2;
    pic_out = (ten265_outpic_t *)malloc(sizeof(ten265_outpic_t));

    /* Get h265 headers */
    FAIL_IF_ERROR(ten265_encoder_header(enc, &nal, &i_nal) < 0,
        "ten265_encoder_open failed.\n");
    for (i = 0; i < i_nal; i++) {
        if (!fwrite(nal[i].payload, nal[i].i_payload, 1, file_out))
            goto fail;
    }

    /* Encode frames */
    for (;; i_frame++) {
        /* Read input frame */
        if (fread(pic.plane[0], 1, luma_size, file_in) != (unsigned)luma_size)
            break;
        if (fread(pic.plane[1], 1, chroma_size , file_in) != (unsigned)chroma_size)
            break;
        if (fread(pic.plane[2], 1, chroma_size , file_in) != (unsigned)chroma_size)
            break;

        /* Update pts */
        pic.i_pts = i_frame;

        /* Reconfig example */
        if (pic.i_pts == 400 && strcmp(extra_func, "reconfig") == 0) {
            if (strcmp(rc_method, "vbv_crf") == 0) {
                params->m_iRateControlMode = TEN265_RC_CRF;
                FAIL_IF_ERROR(ten265_param_parse(params, "crf", "40") < 0, 
                        "ten265_param_parse crf failed.\n");
                params->m_vbvMaxBitrate = 400;
                params->m_vbvBufferSize = 400;
            } else if (strcmp(rc_method, "crf") == 0) {
                params->m_iRateControlMode = TEN265_RC_CRF;
                FAIL_IF_ERROR(ten265_param_parse(params, "crf", "40") < 0, 
                        "ten265_param_parse crf failed.\n");
            }
            FAIL_IF_ERROR(ten265_encoder_reconfig(enc, params) < 0, 
                "ten265_encoder_reconfig failed.\n");
        }

        /* Encode frame */
        ret = ten265_encoder_encode(enc, &pic, pic_out, &extra);
        if (!ret)
            continue;
        if (ret < 0)
            goto fail;
        for (i = 0; i < pic_out->i_nal; i++) {
            if (!fwrite(pic_out->nal[i].payload, pic_out->nal[i].i_payload, 1, file_out))
                goto fail;
        }
    }
    
    /* Flush delayed frames */
    while (ten265_encoder_encode(enc, NULL, pic_out, &extra) > 0) {
        for (i = 0; i < pic_out->i_nal; i++) {
            if (!fwrite(pic_out->nal[i].payload, pic_out->nal[i].i_payload, 1, file_out))
                goto fail;
        }
    }
    ten265_encoder_close(enc, &extra);
    return 0;

fail:
    if (enc) {
        ten265_encoder_close(enc, &extra);
        free(pic.plane[0]);
        free(pic.plane[1]);
        free(pic.plane[2]);
        free(pic_out);
    }
    return -1;
}
