// 编译命令
// gcc -o ten264_example ten264_example.c -I ../lib/ -L ../lib/ -ltscsdk_center -ldl -lpthread -lm -lz
// 运行方法
// ten264_example input.yuv output.h264 [rate_control_method: crf or vbv_crf] [extra_func: subjective or reconfig]
// 例如
// ten264_example input.yuv output.h264 crf

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <libtscsdk.h>

#define FAIL_IF_ERROR(cond, ...)\
do {\
    if ((cond)) {\
        fprintf(stderr, __VA_ARGS__);\
        goto fail;\
    }\
} while (0)

void ten264_log_default(void *p_unused, int i_level, const char *psz_fmt, va_list arg) {
    static const char *psz_prefix[] = {
        [TEN264_LOG_NONE+1]    = "none",
        [TEN264_LOG_ERROR+1]   = "error",
        [TEN264_LOG_WARNING+1] = "warning",
        [TEN264_LOG_INFO+1]    = "info",
        [TEN264_LOG_DEBUG+1]   = "debug"
    };
    fprintf(stderr, "ten264 [%s]: ", psz_prefix[i_level+1]);
    vfprintf(stderr, psz_fmt, arg);
}


int main(int argc, char **argv) {
    FAIL_IF_ERROR(argc < 3, "Example usage: \n"
        "ten264_example input.yuv output.h264 "
        "[rate_control_method: crf or vbv_crf] "
        "[extra_func: subjective or reconfig] \n");
    char *extra_func = "";
    char *rc_method = "crf";
    int i;
    for (i = 3; i < argc; i++) {
        if (strcmp(argv[i], "subjective") == 0 || strcmp(argv[i], "reconfig") == 0)
            extra_func = argv[i];
        if (strcmp(argv[i], "crf") == 0 || strcmp(argv[i], "vbv_crf") == 0)
            rc_method = argv[i];
    }

    ten264_param_t params;
    ten264_picture_t pic;
    ten264_picture_t pic_out;
    ten264_t *enc;
    int i_frame = 0;
    int i_frame_size;
    ten264_nal_t *nal;
    int i_nal;
    void *extra;
    FILE *file_in, *file_out;
    int width = 1920, height = 1080;

    FAIL_IF_ERROR((file_in = fopen(argv[1], "rb")) == NULL, "open input file failed.\n");
    FAIL_IF_ERROR((file_out = fopen(argv[2], "wb")) == NULL, "open output file failed.\n");

    /* Configure params */
    FAIL_IF_ERROR(ten264_param_default_preset(&params, "ultrafast") < 0,
        "ten264_param_default_preset failed.\n");
    params.i_width  = width;
    params.i_height = height;
    params.i_bitdepth = 8;
    params.i_csp = TEN264_CSP_NV12;
    params.b_vfr_input = 0;
    params.b_repeat_headers = 0;

    params.pf_log = ten264_log_default;
    params.i_log_level = TEN264_LOG_DEBUG;

    params.i_timebase_den = 24;
    params.i_timebase_num = 1;
    params.i_fps_num = 24;
    params.i_fps_den = 1;

    // crf参数可选： 21.5, 26.6, 31.7, 36
    if (strcmp(rc_method, "vbv_crf") == 0) {
        params.rc.i_rc_method = TEN264_RC_CRF;
        FAIL_IF_ERROR(ten264_param_parse(&params, "crf", "21.5") < 0, 
                "ten264_param_parse crf failed.\n");
        params.rc.i_vbv_max_bitrate = 400;
        params.rc.i_vbv_buffer_size = 400;
    } else if (strcmp(rc_method, "crf") == 0) {
        params.rc.i_rc_method = TEN264_RC_CRF;
        FAIL_IF_ERROR(ten264_param_parse(&params, "crf", "21.5") < 0, 
                "ten264_param_parse crf failed.\n");
    }

    const char ten264_param[] = "threads=4:keyint=infinite:p-opt=1,0,4,1.5:tune=zerolatency_shxf";
    int ten264_param_len = strlen(ten264_param);
    char key[128] = {0};
    char val[128] = {0};
    int offset = 0;
    while (offset < ten264_param_len) {
        if (sscanf(ten264_param + offset, "%127[^:=]=%127[^:]", key, val) == 2) {
            FAIL_IF_ERROR(ten264_param_parse(&params, key, val) < 0, 
                "ten264_param_parse failed. key=%s, val=%s\n", key, val);
        }
        while (ten264_param[offset] != ':' && offset < ten264_param_len)
            offset ++;
        offset++;
    }

    FAIL_IF_ERROR(!(enc = ten264_encoder_open(&params, &extra, "../sdk_config")),
        "ten264_encoder_open failed.\n");

    if (strcmp(extra_func, "subjective") == 0) {
        FAIL_IF_ERROR(add_preprocess(&extra, "sharp_size=3:sharp_amount=0.5") != 0,
            "add_preprocess failed.\n");
    }

    /* Configure input picture */
    int luma_size = width * height;     // Y
    int chroma_size = luma_size / 2;    // UV

    ten264_picture_init(&pic);
    pic.img.i_plane = 2;
    pic.img.plane[0] = malloc(luma_size);   // Y
    pic.img.plane[1] = malloc(chroma_size); // UV
    pic.img.i_stride[0] = width;
    pic.img.i_stride[1] = width;
    pic.img.i_csp = params.i_csp;

    /* Get h264 headers */
    i_frame_size = ten264_encoder_headers(enc, &nal, &i_nal);
    if (i_frame_size) {
        if (!fwrite(nal->p_payload, i_frame_size, 1, file_out))
            goto fail;
    }

    /* Encode frames */
    for (;; i_frame++) {
        /* Read input frame */
        if (fread(pic.img.plane[0], 1, luma_size, file_in) != (unsigned)luma_size)
            break;
        if (fread(pic.img.plane[1], 1, chroma_size , file_in) != (unsigned)chroma_size)
            break;

        /* Update pts */
        pic.i_pts = i_frame;

        /* Reconfig example */
        if (pic.i_pts == 400 && strcmp(extra_func, "reconfig") == 0) {
            if (strcmp(rc_method, "vbv_crf") == 0) {
                params.rc.i_rc_method = TEN264_RC_CRF;
                FAIL_IF_ERROR(ten264_param_parse(&params, "crf", "36") < 0, 
                        "ten264_param_parse crf failed.\n");
                params.rc.i_vbv_max_bitrate = 200;
                params.rc.i_vbv_buffer_size = 200;
            } else if (strcmp(rc_method, "crf") == 0) {
                params.rc.i_rc_method = TEN264_RC_CRF;
                FAIL_IF_ERROR(ten264_param_parse(&params, "crf", "36") < 0, 
                        "ten264_param_parse crf failed.\n");
            }
            FAIL_IF_ERROR(ten264_encoder_reconfig(enc, &params) < 0, 
                "ten264_encoder_reconfig failed.\n");
        }

        /* Encode frame */
        i_frame_size = ten264_encoder_encode(enc, &nal, &i_nal, &pic, &pic_out, NULL, 0, 0, &extra);
        if (i_frame_size < 0)
            goto fail;
        else if (i_frame_size) {
            if (!fwrite(nal->p_payload, i_frame_size, 1, file_out))
                goto fail;
        }
    }

    /* Flush delayed frames */
    while (ten264_encoder_delayed_frames(enc)) {
        i_frame_size = ten264_encoder_encode(enc, &nal, &i_nal, NULL, &pic_out, NULL, 0, 0, &extra);
        if (i_frame_size < 0)
            goto fail;
        else if (i_frame_size) {
            if (!fwrite(nal->p_payload, i_frame_size, 1, file_out))
                goto fail;
        }
    }
    ten264_encoder_close(enc, &extra);
    return 0;

fail:
    if (enc) {
        ten264_encoder_close(enc, &extra);
        free(pic.img.plane[0]);
        free(pic.img.plane[1]);
    }
    return -1;
}
