#include "libavutil/opt.h"
#include "libavutil/imgutils.h"
#include "libavutil/avassert.h"
#include "avfilter.h"
#include "formats.h"
#include "internal.h"
#include "video.h"
#include "libavutil/pixdesc.h"
#include "libswscale/swscale.h"
#include "libavutil/time.h"
#include <pthread.h>
#define ENABEL_AI
#include <sys/time.h>
#include <time.h>
#include <math.h>

#ifdef ENABEL_AI
#include "libtscsdk.h"
#endif
#define TEN_FILTER_VERSION 55555100

typedef struct HDR2SDRContext {
    wxmmhdr_frame_translator_t *frame_translator;
    int o_trc;
    int o_range;
    int o_pixfmt;
    int skip_handle;
} HDR2SDRContext;

typedef struct TveFilterContext {
    const AVClass *class;
    char* mdir; // the dir of model
    char* af_option;
    char* sr_option;
    char* sp_option; //sharpness
    char* gpu_list;
    char* log_level;

    float bright_enhance;
	
    int face_protect_enable;
    float face_af_ratio;
    float face_sp_ratio;
    float global_sp_ratio;
    float global_af_ratio;
    float vquality;
    int size;
    float amount;
    float ratio;
    int type;

    int tve_has_init;
    int gpu;
    int ctx_id;
    //add some private data if you want
    int in_frame_h, in_frame_w, out_frame_h, out_frame_w;
    int hsub0,vsub0;
    int filter_type;
    int buff_size;
    uint8_t * video_yuv_buf;
#ifdef ENABEL_AI
    TENParam tve_param;
#endif

    struct timeval tv_start;

    // thread async
    pthread_t tve_thread;
	pthread_mutex_t tve_lock;
	pthread_cond_t tve_cond_in;
	pthread_cond_t tve_cond_out;
	int tve_thread_run;
	int tve_done;
	int tve_ret;
	AVFilterContext *tve_dst;
	AVFrame *tve_fr_in;
	AVFrame *tve_fr_out;
	uint8_t * tve_current_buffer;
	int tve_index_buffer;
	int first_frame;

    // add pingpong
    int (* apply_video_enhance)(AVFilterContext *ctx, AVFrame *dst, AVFrame *src, int gpu_device, uint8_t* yuv_buff);
	pthread_t unsharp_ping_thread;
	pthread_mutex_t ping_lock;
	pthread_cond_t ping_in_cond;
	pthread_cond_t ping_out_cond;
	AVFilterContext *ping_dst;
	AVFrame *ping_in;
	AVFrame *ping_out;
	uint8_t * ping_yuv_buf;
	int ping_run;
	int ping_done;
	int ping_ret;
	int ping_gpu;
	int ping_ctx_id;

	pthread_t unsharp_pong_thread;
	pthread_mutex_t pong_lock;
	pthread_cond_t pong_in_cond;
	pthread_cond_t pong_out_cond;
	AVFilterContext *pong_dst;
	AVFrame *pong_in;
	AVFrame *pong_out;
	uint8_t * pong_yuv_buf;
	int pong_run;
	int pong_done;
	int pong_ret;
	int pong_gpu;
	int pong_ctx_id;
	int pingpong;

	int enable_tvenai_context;
	int hdr2sdr;
	UnsharpData ud;
	HDR2SDRContext hdr2sdr_ctx;
} TveFilterContext;

typedef struct ThreadData {		
    uint16_t *in;	
    uint16_t *out;
} ThreadData;


static inline int from_AVColorPrimaries(int pri) {
  switch (pri) {
    case AVCOL_PRI_BT2020:
      return 0;
    case AVCOL_PRI_BT709:
    case AVCOL_PRI_UNSPECIFIED:
      return 1;
    default:
      return -1;
  }
}

static inline int from_AVColorTransferCharacteristic(int trc) {
  switch (trc) {
    case AVCOL_TRC_ARIB_STD_B67:
      return 2;
    case AVCOL_TRC_SMPTE2084:
      return 1;
    case AVCOL_TRC_BT709:
    case AVCOL_TRC_UNSPECIFIED:
      return 0;
    default:
      return -1;
  }
}

static inline int from_AVColorSpace(int spc) {
  switch (spc) {
    case AVCOL_SPC_BT2020_NCL:
      return 1;
    case AVCOL_SPC_BT709:
    case AVCOL_SPC_UNSPECIFIED:
      return 4;
    case AVCOL_SPC_RGB:
      return 0;
    case AVCOL_SPC_BT2020_CL:
      return 2;
    case AVCOL_SPC_ICTCP:
      return 3;
    default:
      return -1;
  }
}

static inline int from_AVColorRange(int range) {
  if (range == AVCOL_RANGE_JPEG)
    return 1;
  else
    return 0;
}

static inline int from_AVPixelFormat(int pixfmt) {
  switch (pixfmt) {
    case AV_PIX_FMT_YUV420P:
    case AV_PIX_FMT_YUVJ420P:
      return 0;
    case AV_PIX_FMT_YUV420P10:
      return 1;
    case AV_PIX_FMT_RGBA:
      return 2;
    default:
      return -1;
  }
}

static av_cold int init_Naitve(AVFilterContext *ctx)
{
    TveFilterContext *privCtx = ctx->priv;

    if (privCtx->type && privCtx->amount <= 0)
    {
        privCtx->size = 7;
        privCtx->amount = 0.5;
        privCtx->ratio = 0.98;
    }

    if (privCtx->vquality > 0 && privCtx->amount > 0)
    {
        privCtx->amount = adapt_sharp_filter(privCtx->vquality);
    }

    if (privCtx->amount > 0)
    {
        privCtx->enable_tvenai_context = 1;
        if (privCtx->size %2 == 0)
            privCtx->size++;

        privCtx->ud.size = privCtx->size;
        privCtx->ud.amount = privCtx->amount;
        privCtx->ud.ratio = privCtx->ratio;
        privCtx->ud.sr = NULL;
        privCtx->ud.sc = NULL;
        privCtx->ud.extra = NULL;
    }

    if (privCtx->hdr2sdr > 0)
    {
        privCtx->enable_tvenai_context = 1;
        privCtx->hdr2sdr_ctx.frame_translator = NULL;
        privCtx->hdr2sdr_ctx.o_trc = 0;
        privCtx->hdr2sdr_ctx.o_range = 0;
        privCtx->hdr2sdr_ctx.o_pixfmt = 0;
        privCtx->hdr2sdr_ctx.skip_handle = 0;
    }
    return 0;
}

static int executeNaiTVE(AVFilterContext *ctx, AVFrame *out, AVFrame *in)
{
    AVFilterLink *inlink = ctx->inputs[0];
    TveFilterContext *privCtx = ctx->priv;

    if (privCtx->amount > 0.01f)
    {
        privCtx->ud.dst = out->data[0];
        privCtx->ud.src =  in->data[0];
        privCtx->ud.width = inlink->w;
        privCtx->ud.height = inlink->h;
        privCtx->ud.dst_stride = out->linesize[0];
        privCtx->ud.src_stride = in->linesize[0];
        if (unsharp_slice(&privCtx->ud) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "unsharp slice failed, use copy %d! \n", av_frame_copy(out, in));
        }
    }

    if (privCtx->hdr2sdr > 0)
    {
        HDR2SDRContext *s = &privCtx->hdr2sdr_ctx;
        const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get(inlink->format);
        int inputBitDepth = desc->comp[0].depth;

        uint8_t * src[3];
        uint8_t * dst[3];
        long srcStride[3], dstStride[3];

        srcStride[0] = in->linesize[0];
        srcStride[1] = in->linesize[1];
        srcStride[2] = in->linesize[2];
        dstStride[0] = out->linesize[0];
        dstStride[1] = out->linesize[1];
        dstStride[2] = out->linesize[2];
        src[0] = in->data[0];
        src[1] = in->data[1];
        src[2] = in->data[2];
        dst[0] = out->data[0];
        dst[1] = out->data[1];
        dst[2] = out->data[2];

        if (s->skip_handle > 0)
        {
            av_frame_copy(dst , in);
            return 0;
        }

        if (!s->frame_translator) {
            wxmmhdr_init_param_t init_param = {
                .i = {
                    .pri = from_AVColorPrimaries(in->color_primaries),
                    .trc = from_AVColorTransferCharacteristic(in->color_trc),
                    .spc = from_AVColorSpace(in->colorspace),
                    .range = from_AVColorRange(in->color_range),
                    .fmt = from_AVPixelFormat(inlink->format),
                },
                .o = {
                    .pri = from_AVColorPrimaries(AVCOL_PRI_BT709),
                    .trc = from_AVColorTransferCharacteristic(AVCOL_TRC_BT709),
                    .spc = from_AVColorSpace(AVCOL_SPC_BT709),
                    .range = from_AVColorRange(AVCOL_RANGE_MPEG),
                    .fmt = from_AVPixelFormat(AV_PIX_FMT_YUV420P),
                },
                .use_threads = 8,
                .use_assembly = 1,
                .use_cuda = 1,
            };
            s->frame_translator = wxmmhdr_open(init_param);
            if (!s->frame_translator) {
                av_frame_copy(dst , in);
                s->skip_handle = 1;
                return 0;
            }
        }
        wxmmhdr_process(s->frame_translator, src, srcStride, dst, dstStride, in->width, in->height);
    }
    return 0;
}

static int frame_enhance_video(AVFilterContext *ctx, AVFrame *dst, AVFrame *src, int ctx_id, uint8_t* yuv_buff)
{
	TveFilterContext *privCtx = ctx->priv;
    if (dst->width  < src->width ||
        dst->height < src->height)
        return AVERROR(EINVAL);

    int i, planes;
    planes = av_pix_fmt_count_planes(dst->format);
    //make sure data is valid
    for (i = 0; i < planes; i++)
        if (!dst->data[i] || !src->data[i])
            return AVERROR(EINVAL);

    // get yuv buffer, maybe takes 1ms?
    int ret = av_image_copy_to_buffer(yuv_buff, privCtx->buff_size, src->data, src->linesize, src->format, src->width, src->height, 1);
    if(ret < 0) {
    	av_log(NULL, AV_LOG_ERROR, "av_image_copy_to_buffer fail, ret:%d\n",ret);
    	return ret;
    }

    if (privCtx->gpu_list != NULL)
    {
        ret =  executeTEN(yuv_buff, src->height, src->width, 3, yuv_buff, privCtx->buff_size, ctx_id);
        if(ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "executeTEN fail, ret:%d\n",ret);
            return ret;
        }
    }

	// copy tve to out
	int uv_w = FF_CEIL_RSHIFT(privCtx->out_frame_w, privCtx->hsub0);
	int uv_h = FF_CEIL_RSHIFT(privCtx->out_frame_h, privCtx->vsub0);

	//copy y from buffer to out
	int plane=0; // y
	uint8_t *tmp = dst->data[plane];
	uint8_t *tmp_predict = yuv_buff;
	for (i = 0; i < privCtx->out_frame_h; i++){
	   memcpy(tmp, tmp_predict, privCtx->out_frame_w*sizeof(uint8_t));
	   tmp_predict += privCtx->out_frame_w;
	   tmp += dst->linesize[plane];
	}

	//copy u
	tmp = dst->data[plane+1];
	for (i = 0; i < uv_h; i++){
	   memcpy(tmp, tmp_predict, uv_w*sizeof(uint8_t));
	   tmp_predict += uv_w;
	   tmp += dst->linesize[plane+1];
	}

	//copy v
	tmp = dst->data[plane+2];
	for (i = 0; i < uv_h; i++){
	   memcpy(tmp, tmp_predict, uv_w*sizeof(uint8_t));
	   tmp_predict += uv_w;
	   tmp += dst->linesize[plane+2];
	}

    return 0;
}

static void *call_ping_video_enhance(void *ctx)
{
	TveFilterContext *s = (TveFilterContext *)ctx;
    while(s->ping_run) {
        pthread_mutex_lock(&s->ping_lock);
        while(s->ping_in == NULL && s->ping_run) {
            pthread_cond_wait(&s->ping_in_cond, &s->ping_lock);
        }

        if(s->ping_run) {
        	s->ping_ret = s->apply_video_enhance(s->ping_dst, s->ping_out, s->ping_in, s->ping_ctx_id, s->ping_yuv_buf);
            s->ping_done = 1;
            pthread_cond_signal(&s->ping_out_cond);
        }

        pthread_mutex_unlock(&s->ping_lock);
        av_usleep(1);
    }

    return ((void*)0);
}

static void *call_pong_video_enhance(void *ctx)
{
	TveFilterContext *s = (TveFilterContext *)ctx;
    while(s->pong_run) {
        pthread_mutex_lock(&s->pong_lock);
        while(s->pong_in == NULL && s->pong_run) {
            pthread_cond_wait(&s->pong_in_cond, &s->pong_lock);
        }

        if(s->pong_run) {
        	s->pong_ret = s->apply_video_enhance(s->pong_dst, s->pong_out, s->pong_in, s->pong_ctx_id, s->pong_yuv_buf);
            s->pong_done = 1;
            pthread_cond_signal(&s->pong_out_cond);
        }

        pthread_mutex_unlock(&s->pong_lock);
        av_usleep(1);
    }

    return ((void*)0);
}

static void *tve_run(void *ctx)
{
	TveFilterContext *s = (TveFilterContext *)ctx;
	av_log(NULL, AV_LOG_ERROR, "tve thread runnning\n");
    while(s->tve_thread_run) {
        pthread_mutex_lock(&s->tve_lock);
        while((s->tve_done == 1) && s->tve_thread_run) {
            pthread_cond_wait(&s->tve_cond_in, &s->tve_lock);
        }
        if(s->tve_thread_run && s->tve_fr_in != NULL) {
        	s->tve_ret = s->apply_video_enhance(s->tve_dst, s->tve_fr_out, s->tve_fr_in, s->ctx_id, s->tve_current_buffer);
            s->tve_done = 1;
            pthread_cond_signal(&s->tve_cond_out);
        }
        pthread_mutex_unlock(&s->tve_lock);
        av_usleep(1);
    }

    return ((void*)0);
}

static int filter_frame(AVFilterLink *link, AVFrame *in)
{
    AVFilterContext *avctx = link->dst;
    AVFilterLink *outlink = avctx->outputs[0];
    TveFilterContext *privCtx = avctx->priv;
    AVFrame *out;
    int ret=0;

    //allocate a new buffer, data is null
    out = ff_get_video_buffer(outlink, outlink->w, outlink->h);
    if (!out) {
        av_frame_free(&in);
        return AVERROR(ENOMEM);
    }
    //the new output frame, property is the same as input frame, only width/height is different
    av_frame_copy_props(out, in);

    // sr will be change w and h
    out->width  = outlink->w;
    out->height = outlink->h;
    privCtx->hsub0 = av_pix_fmt_desc_get(outlink->format)->log2_chroma_w;
    privCtx->vsub0 = av_pix_fmt_desc_get(outlink->format)->log2_chroma_h;

    if(!(privCtx->ping_gpu>=0 && privCtx->pong_gpu>=0)) {
        ret = frame_enhance_video(avctx, out, in, privCtx->ctx_id, privCtx->video_yuv_buf);
        if(ret) goto END;
    } else {
    	if(privCtx->pingpong % 2 == 0) {
			pthread_mutex_lock(&privCtx->ping_lock);
			privCtx->ping_in = in;
			privCtx->ping_out = out;
			privCtx->ping_dst = link->dst;
			pthread_mutex_unlock(&privCtx->ping_lock);
			pthread_cond_signal(&privCtx->ping_in_cond);
		}
		else {
			pthread_mutex_lock(&privCtx->pong_lock);
			privCtx->pong_in = in;
			privCtx->pong_out = out;
			privCtx->pong_dst = link->dst;
			pthread_mutex_unlock(&privCtx->pong_lock);
			pthread_cond_signal(&privCtx->pong_in_cond);
		}
    	// get frame
		if(privCtx->pingpong % 2 == 1 && privCtx->pingpong > 0) {
			pthread_mutex_lock(&privCtx->ping_lock);
			while(privCtx->ping_done == 0){
				pthread_cond_wait(&privCtx->ping_out_cond, &privCtx->ping_lock);
			}
			in = privCtx->ping_in;
			out = privCtx->ping_out;
			privCtx->ping_in = NULL;
			privCtx->ping_out = NULL;
			privCtx->ping_done = 0;
			ret = privCtx->ping_ret;
			pthread_mutex_unlock(&privCtx->ping_lock);
		}
		else if(privCtx->pingpong % 2 == 0 && privCtx->pingpong > 0) {
			pthread_mutex_lock(&privCtx->pong_lock);
			while(privCtx->pong_done == 0){
				pthread_cond_wait(&privCtx->pong_out_cond, &privCtx->pong_lock);
			}
			in = privCtx->pong_in;
			out = privCtx->pong_out;
			privCtx->pong_in = NULL;
			privCtx->pong_out = NULL;
			privCtx->pong_done = 0;
			ret = privCtx->pong_ret;
			pthread_mutex_unlock(&privCtx->pong_lock);
		}
		else{
			privCtx->pingpong ++;
			return 0;// ff_filter_frame(outlink, NULL);
		}
		privCtx->pingpong = 3 - privCtx->pingpong;
    }

    if(privCtx->enable_tvenai_context > 0) {
        ret = executeNaiTVE(avctx, out, in);
        if(ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "executeNaiTVE fail, ret:%d\n",ret);
            goto END;
        }
    }

END:
    av_frame_free(&in);
    if (ret < 0) {
		av_frame_free(&out);
		return ret;
	}
    return ff_filter_frame(outlink, out);
}

static int request_frame(AVFilterLink *outlink)
{
    AVFilterContext *ctx = outlink->src;
    TveFilterContext        *privCtx = ctx->priv;
    int ret;

    ret = ff_request_frame(ctx->inputs[0]);

    /* flush the fifo */
    if(!(privCtx->ping_gpu>=0 && privCtx->pong_gpu>=0) && ret == AVERROR_EOF && privCtx->tve_fr_out != NULL) { // only one gpu
    	//wait
		pthread_mutex_lock(&privCtx->tve_lock);
		while(privCtx->tve_done == 0){
			pthread_cond_wait(&privCtx->tve_cond_out, &privCtx->tve_lock);
		}

		AVFrame *temp_in = privCtx->tve_fr_in;
		AVFrame *temp_out = privCtx->tve_fr_out;
		privCtx->tve_thread_run = 0;
		privCtx->tve_fr_in = NULL;
		privCtx->tve_fr_out = NULL;
		privCtx->tve_done = 0;
		ret = privCtx->tve_ret;
		pthread_mutex_unlock(&privCtx->tve_lock);
		pthread_cond_signal(&privCtx->tve_cond_in);

		if(ret) {
			return ret;
		}

		av_frame_free(&temp_in);
		ret = ff_filter_frame(outlink, temp_out);
		if(ret != 0) {
			return ret;
		}

		return 0;
    }
    else if(ret == AVERROR_EOF && privCtx->pingpong % 2 == 1 && privCtx->pingpong > 0 && privCtx->ping_out != NULL) {
		pthread_mutex_lock(&privCtx->ping_lock);
		while(privCtx->ping_done == 0){
			pthread_cond_wait(&privCtx->ping_out_cond, &privCtx->ping_lock);
		}
		int ret = privCtx->ping_ret;
		if(ret != 0) {
			return ret;
		}

		if(privCtx->enable_tvenai_context > 0) {
			// color
			ret = executeNaiTVE(ctx, privCtx->ping_out, privCtx->ping_out);
			if(ret < 0) {
				av_log(NULL, AV_LOG_ERROR, "executeNaiTVE fail, ret:%d\n",ret);
				return ret;
			}
		}


		ret = ff_filter_frame(outlink, privCtx->ping_out);
		if(ret != 0) {
			return ret;
		}
		av_frame_free(&privCtx->ping_in);
		privCtx->ping_in = NULL;
		privCtx->ping_out = NULL;
		privCtx->ping_done = 0;
		pthread_mutex_unlock(&privCtx->ping_lock);
		return 0;
	}
	else if(ret == AVERROR_EOF && privCtx->pingpong % 2 == 0 && privCtx->pingpong > 0 && privCtx->pong_out != NULL) {
		pthread_mutex_lock(&privCtx->pong_lock);
		while(privCtx->pong_done == 0){
			pthread_cond_wait(&privCtx->pong_out_cond, &privCtx->pong_lock);
		}
		int ret = privCtx->pong_ret;
		if(ret != 0) {
			return ret;
		}


		ret = ff_filter_frame(outlink, privCtx->pong_out);
		if(ret != 0) {
			return ret;
		}
		av_frame_free(&privCtx->pong_in);
		privCtx->pong_in = NULL;
		privCtx->pong_out = NULL;
		privCtx->pong_done = 0;
		pthread_mutex_unlock(&privCtx->pong_lock);
		return 0;
	}

    return ret;
}

static av_cold int config_output(AVFilterLink *outlink)
{
    AVFilterContext *ctx = outlink->src;
    TveFilterContext *privCtx = ctx->priv;

    outlink->w=privCtx->out_frame_w;
    outlink->h=privCtx->out_frame_h;

    av_log(NULL, AV_LOG_DEBUG, "configure output, w h = (%d %d), format %d \n", outlink->w, outlink->h, outlink->format);

    return 0;
}

static int config_input(AVFilterLink *link)
{
	TveFilterContext *privCtx = link->dst->priv;

	privCtx->in_frame_h = link->h;
	privCtx->in_frame_w = link->w;

	privCtx->out_frame_h = privCtx->in_frame_h;
	privCtx->out_frame_w = privCtx->in_frame_w;

#ifdef ENABEL_AI
	if (privCtx->gpu_list != NULL)
	{
	    if(privCtx->filter_type&TEN_FILTER_SR_MODE_X2) {
	        privCtx->out_frame_h *= 2;
	        privCtx->out_frame_w *= 2;
	    } else if(privCtx->filter_type&TEN_FILTER_SR_MODE_X3) {
	        privCtx->out_frame_h *= 3;
	        privCtx->out_frame_w *= 3;
	    }
	}
#endif
	privCtx->buff_size = av_image_get_buffer_size(AV_PIX_FMT_RGB24, privCtx->out_frame_w, privCtx->out_frame_h, 1);

	if(privCtx->video_yuv_buf == NULL) {
		privCtx->video_yuv_buf = ( uint8_t *)calloc(privCtx->buff_size, sizeof(uint8_t)); // max rgb
		if(privCtx->video_yuv_buf == NULL) {
			av_log(NULL, AV_LOG_ERROR, "### alloc buff fail\n");
			return -5;
		}
	}
	if(privCtx->pong_yuv_buf == NULL) {
		privCtx->pong_yuv_buf = ( uint8_t *)calloc(privCtx->buff_size, sizeof(uint8_t)); // max rgb
		if(privCtx->pong_yuv_buf == NULL) {
			av_log(NULL, AV_LOG_ERROR, "### alloc pong buff fail\n");
			return -6;
		}
	}

	if(privCtx->ping_yuv_buf == NULL) {
		privCtx->ping_yuv_buf = ( uint8_t *)calloc(privCtx->buff_size, sizeof(uint8_t)); // max rgb
		if(privCtx->ping_yuv_buf == NULL) {
			av_log(NULL, AV_LOG_ERROR, "### alloc ping buff fail\n");
			return -6;
		}
	}

#ifdef ENABEL_AI

	if (privCtx->gpu_list != NULL)
	{
	    if(!(privCtx->ping_gpu>=0 && privCtx->pong_gpu>=0)) {
	        int ret = initTEN(privCtx->in_frame_h, privCtx->in_frame_w, privCtx->filter_type, privCtx->mdir, &privCtx->tve_param, privCtx->gpu, &privCtx->ctx_id);
	        if(ret != 0) {
	            av_log(NULL, AV_LOG_ERROR, "init tve fail, ret:%d\n",ret);
	            return ret;
	        }
	    } else {
	        int ret = initTEN(privCtx->in_frame_h, privCtx->in_frame_w, privCtx->filter_type, privCtx->mdir, &privCtx->tve_param, privCtx->ping_gpu, &privCtx->ping_ctx_id);
	        if(ret != 0) {
	            av_log(NULL, AV_LOG_ERROR, "init ping tve fail, ret:%d\n",ret);
	            return ret;
	        }

	        ret = initTEN(privCtx->in_frame_h, privCtx->in_frame_w, privCtx->filter_type, privCtx->mdir, &privCtx->tve_param, privCtx->pong_gpu, &privCtx->pong_ctx_id);
	        if(ret != 0) {
	            av_log(NULL, AV_LOG_ERROR, "init pong tve fail, ret:%d\n",ret);
	            return ret;
	        }
	    }
	    privCtx->tve_has_init = 1;
	}
#endif

    return 0;
}

static av_cold int init(AVFilterContext *ctx)
{
    TveFilterContext *privCtx = ctx->priv;
    privCtx->tve_has_init = -1;
    int ret=0;
    gettimeofday(&privCtx->tv_start,NULL);

    privCtx->enable_tvenai_context=0;
    init_Naitve(ctx);
    av_log(NULL, AV_LOG_INFO, "privCtx->param:face_protect_enable:%d, face_af_ratio:%.2f,face_sp_ratio:%.2f,global_sp_ratio:%.2f,global_af_ratio:%.2f, sharp %d %f %f \n",
            privCtx->tve_param.face_protect_enable,
            privCtx->tve_param.face_af_ratio,
            privCtx->tve_param.face_sp_ratio,
            privCtx->tve_param.global_sp_ratio,
            privCtx->tve_param.reserve_param_float[0],
            privCtx->size,
            privCtx->amount,
            privCtx->ratio);

    if (privCtx->gpu_list != NULL)
    {
        initTenParam(&privCtx->tve_param); // init global
        privCtx->tve_param.face_protect_enable = privCtx->face_protect_enable;
        privCtx->tve_param.face_af_ratio = privCtx->face_af_ratio;
        privCtx->tve_param.face_sp_ratio = privCtx->face_sp_ratio;
        privCtx->tve_param.global_sp_ratio = privCtx->global_sp_ratio;
        privCtx->tve_param.reserve_param_float[0] = privCtx->global_af_ratio;
        av_log(NULL, AV_LOG_INFO, "privCtx->param:face_protect_enable:%d, face_af_ratio:%.2f,face_sp_ratio:%.2f,global_sp_ratio:%.2f,global_af_ratio:%.2f,: sharp %d %f %f \n",
                privCtx->tve_param.face_protect_enable,
                privCtx->tve_param.face_af_ratio,
                privCtx->tve_param.face_sp_ratio,
                privCtx->tve_param.global_sp_ratio,
                privCtx->tve_param.reserve_param_float[0],
                privCtx->size,
                privCtx->amount,
                privCtx->ratio);
        privCtx->video_yuv_buf = NULL;

        if(privCtx->af_option != NULL) {
            int type = 0;
            if (privCtx->vquality > 0.0f)
                type = adapt_tve_filter(privCtx->vquality);

            privCtx->filter_type |= TEN_FILTER_AF;
            if(strncmp(privCtx->af_option,"weak", strlen(privCtx->af_option)) == 0 || type ==2)  {
                privCtx->filter_type |= TEN_FILTER_AF_MODE_WEAK;
            } else if(strncmp(privCtx->af_option,"strong", strlen(privCtx->af_option)) == 0) {
                privCtx->filter_type |= TEN_FILTER_AF_MODE_STRONG;
            } else if(strncmp(privCtx->af_option,"auto", strlen(privCtx->af_option)) == 0 || type ==1) {
                privCtx->filter_type |= TEN_FILTER_AF_MODE_AUTO;
            }  else if(strncmp(privCtx->af_option,"average", strlen(privCtx->af_option)) == 0) {
                privCtx->filter_type |= TEN_FILTER_AF_MODE_AVERAGE;
            } else {
                av_log(NULL, AV_LOG_ERROR, "privCtx->af_option:%s must be weak,average,strong,auto\n", privCtx->af_option);
                return -5;
            }
            av_log(NULL, AV_LOG_INFO, "privCtx->af_option:%s\n", privCtx->af_option);
        }

        if(privCtx->log_level != NULL) {
            if(strncmp(privCtx->log_level,"warning", strlen(privCtx->log_level)) == 0) {
                privCtx->filter_type |= TEN_LOG_WARNING;
            } else if(strncmp(privCtx->log_level,"info", strlen(privCtx->log_level)) == 0) {
                privCtx->filter_type |= TEN_LOG_INFO;
            } else if(strncmp(privCtx->log_level,"error", strlen(privCtx->log_level)) == 0) {
                privCtx->filter_type |= TEN_LOG_ERROR;
            } else if(strncmp(privCtx->log_level,"internal_error", strlen(privCtx->log_level)) == 0) {
                privCtx->filter_type |= TEN_LOG_INTERNAL_ERROR;
            }
            av_log(NULL, AV_LOG_INFO, "privCtx->log_level:%s\n", privCtx->log_level);
        }
    }

	privCtx->apply_video_enhance = frame_enhance_video;
	privCtx->gpu = 0;
	privCtx->ping_gpu = -1;
	privCtx->pong_gpu = -1;

	if(privCtx->gpu_list != NULL) {
		int g_ = privCtx->gpu_list[0]-'0';
		if(g_ >=0 && g_ <=9) {
			privCtx->gpu = g_;
			privCtx->ping_gpu = g_;
		}
		if(strlen(privCtx->gpu_list) > 1) {
			g_ = privCtx->gpu_list[1]-'0';
			if(g_ >=0 && g_ <=9) {
				privCtx->pong_gpu = g_;
			}
		}
		av_log(NULL, AV_LOG_INFO, "privCtx->gpu_list:%s\n", privCtx->gpu_list);
	}


	if(privCtx->ping_gpu>=0 && privCtx->pong_gpu>=0) {
		pthread_mutex_init(&privCtx->ping_lock, NULL);
		pthread_cond_init (&privCtx->ping_in_cond, NULL);
		pthread_cond_init (&privCtx->ping_out_cond, NULL);
		privCtx->ping_run = 1;
		privCtx->ping_done = 0;
		privCtx->ping_in = NULL;
		privCtx->ping_out = NULL;
		privCtx->ping_yuv_buf = NULL;
		privCtx->ping_ret = 0;
		ret = pthread_create(&privCtx->unsharp_ping_thread, NULL, call_ping_video_enhance, (void *)privCtx);
		if (ret) {
			av_log(ctx, AV_LOG_ERROR, "Thread ping creation failed.\n");
			return AVERROR(EINVAL);
		}

		pthread_mutex_init(&privCtx->pong_lock, NULL);
		pthread_cond_init (&privCtx->pong_in_cond, NULL);
		pthread_cond_init (&privCtx->pong_out_cond, NULL);
		privCtx->pong_run = 1;
		privCtx->pong_done = 0;
		privCtx->pong_in = NULL;
		privCtx->pong_out = NULL;
		privCtx->pong_yuv_buf = NULL;
		privCtx->pong_ret = 0;
		ret = pthread_create(&privCtx->unsharp_pong_thread, NULL, call_pong_video_enhance, (void *)privCtx);
		if (ret) {
			av_log(ctx, AV_LOG_ERROR, "Thread pong creation failed.\n");
			return AVERROR(EINVAL);
		}

		privCtx->pingpong = 0; // first time
	} else { // thread
		privCtx->tve_thread_run=1;
		privCtx->tve_done=1;
		privCtx->tve_ret=0;
		privCtx->tve_fr_in=NULL;
		privCtx->tve_fr_out=NULL;
		privCtx->first_frame=1;
		privCtx->tve_index_buffer=0;
	}

	privCtx->tve_has_init = 0;
	privCtx->ctx_id = -1;
	privCtx->ping_ctx_id = -1;
	privCtx->pong_ctx_id = -1;

	int opts = 0;
	if (privCtx->gpu_list != NULL)
	    opts++;
	if (privCtx->hdr2sdr > 0)
	    opts++;
	if (privCtx->amount > 0.01f)
	    opts++;

	if (opts > 1)
	{
	    av_log(ctx, AV_LOG_ERROR, "gpu ai enhance / hdr2sdr / unsharp can only be used once in one filter !!! \n");
	    return -6;
	}

	av_log(ctx, AV_LOG_INFO, "init tenfilter success  !!! \n");
    return 0;
}

static av_cold void uninit(AVFilterContext *ctx)
{
	TveFilterContext *privCtx = ctx->priv;
    av_log(NULL, AV_LOG_DEBUG, "uninit \n");

	if(privCtx->ping_gpu>=0 && privCtx->pong_gpu>=0) {
		pthread_mutex_lock(&privCtx->ping_lock);
		privCtx->ping_run = 0;
		pthread_cond_signal(&privCtx->ping_in_cond);
		pthread_mutex_unlock(&privCtx->ping_lock);
		if(privCtx->tve_has_init>=0)
		pthread_join(privCtx->unsharp_ping_thread, NULL);

		pthread_mutex_destroy(&privCtx->ping_lock);
		pthread_cond_destroy(&privCtx->ping_in_cond);
		pthread_cond_destroy(&privCtx->ping_out_cond);
		if(privCtx->ping_yuv_buf) {
			free(privCtx->ping_yuv_buf);
			privCtx->ping_yuv_buf=NULL;
		}

		pthread_mutex_lock(&privCtx->pong_lock);
		privCtx->pong_run = 0;
		pthread_cond_signal(&privCtx->pong_in_cond);
		pthread_mutex_unlock(&privCtx->pong_lock);
		if(privCtx->tve_has_init>=0)
		pthread_join(privCtx->unsharp_pong_thread, NULL);

		pthread_mutex_destroy(&privCtx->pong_lock);
		pthread_cond_destroy(&privCtx->pong_in_cond);
		pthread_cond_destroy(&privCtx->pong_out_cond);
		if(privCtx->pong_yuv_buf) {
			free(privCtx->pong_yuv_buf);
			privCtx->pong_yuv_buf=NULL;
		}

		if(privCtx->tve_has_init == 1 && privCtx->gpu_list != NULL) {
			uinitTEN(privCtx->ping_ctx_id);
			uinitTEN(privCtx->pong_ctx_id);
			privCtx->tve_has_init = 0;
		}
		av_log(NULL, AV_LOG_INFO, "uninit ping pong \n");
	} else {
		if(privCtx->tve_has_init == 1 && privCtx->gpu_list != NULL) {
			uinitTEN(privCtx->ctx_id);
			privCtx->tve_has_init = 0;
		}
	}

	if(privCtx->video_yuv_buf != NULL) {
		free(privCtx->video_yuv_buf);
		privCtx->video_yuv_buf=NULL;
	}

	struct timeval tv_end;
	gettimeofday(&tv_end,NULL);
	long long s_= (tv_end.tv_sec-privCtx->tv_start.tv_sec);
	long long  ms_=(tv_end.tv_usec-privCtx->tv_start.tv_usec)/1000;
	av_log(NULL, AV_LOG_ERROR, "vf_filter(version:%d) total_run_time:%llds\t%lldms\n",TEN_FILTER_VERSION,s_,ms_);
}

//currently we just support the most common YUV420, can add more if needed
#ifndef FFMPEG5_SDK_VERSION
static int query_formats(AVFilterContext *ctx)
{
    static const enum AVPixelFormat pix_fmts[] = {
        AV_PIX_FMT_YUV420P,
        AV_PIX_FMT_YUVJ420P,
        AV_PIX_FMT_NONE
    };
    AVFilterFormats *fmts_list = ff_make_format_list(pix_fmts);
    if (!fmts_list)
        return AVERROR(ENOMEM);
    return ff_set_common_formats(ctx, fmts_list);
}
#else
static const enum AVPixelFormat pix_fmts[] = {
        AV_PIX_FMT_YUV420P,
        AV_PIX_FMT_YUVJ420P,
        AV_PIX_FMT_NONE
};
#endif

//*************
#define OFFSET(x) offsetof(TveFilterContext, x)
#define FLAGS AV_OPT_FLAG_VIDEO_PARAM|AV_OPT_FLAG_FILTERING_PARAM

static const AVOption tvefilter_options[] = {
	{ "mdir", "set dir of model info, such as mdir=./tve",  OFFSET(mdir), AV_OPT_TYPE_STRING, {.str = NULL}, 0, 0, FLAGS },
	{ "af", "set artifact option, such as af=weak/average/strong/auto",  OFFSET(af_option), AV_OPT_TYPE_STRING, {.str = NULL}, 0, 0, FLAGS },
    { "vquality", "set video quality ",OFFSET(vquality),   AV_OPT_TYPE_FLOAT, { .dbl = -1 }, -1, 100, FLAGS },
    { "sr", "set super resolution option, such as sr=2/3",  OFFSET(sr_option), AV_OPT_TYPE_STRING, {.str = NULL}, 0, 0, FLAGS },
    { "sp", "set sharpness option, such as sp=weak/strong",  OFFSET(sp_option), AV_OPT_TYPE_STRING, {.str = NULL}, 0, 0, FLAGS },
    { "log", "set log level option, such as log=info/warning/error",OFFSET(log_level),   AV_OPT_TYPE_STRING, { .str = NULL }, 0, 0, FLAGS },
    { "gpu", "set gpu list option, such as gpu=0/01",OFFSET(gpu_list),   AV_OPT_TYPE_STRING, { .str = NULL }, 0, 0, FLAGS },
	{ "face_protect_enable", "face_protect_enable, such as face_protect_enable=0/1",OFFSET(face_protect_enable),   AV_OPT_TYPE_INT, { .i64 = 0 }, 0, 3, FLAGS },
    { "face_af_ratio", "face_af_ratio, such as face_af_ratio=0.5",OFFSET(face_af_ratio),   AV_OPT_TYPE_FLOAT, { .dbl = 0.6 }, 0, 1, FLAGS },
    { "face_sp_ratio", "face_sp_ratio, such as face_sp_ratio=0.5",OFFSET(face_sp_ratio),   AV_OPT_TYPE_FLOAT, { .dbl = 1.3 }, 0, 2, FLAGS },
    { "global_sp_ratio", "global_sp_ratio, such as global_sp_ratio=0.5",OFFSET(global_sp_ratio),   AV_OPT_TYPE_FLOAT, { .dbl = 1 }, 0, 2, FLAGS },
    { "global_af_ratio", "global_af_ratio, such as global_af_ratio=0.5",OFFSET(global_af_ratio),   AV_OPT_TYPE_FLOAT, { .dbl = 0.5 }, 0, 2, FLAGS },
    { "unsharp_size", "",OFFSET(size),   AV_OPT_TYPE_INT, { .i64 = 3 }, 3, 11, FLAGS },
    { "unsharp_amount", "",OFFSET(amount),   AV_OPT_TYPE_FLOAT, { .dbl = 0 }, 0, 2, FLAGS },
    { "unsharp_ratio", "",OFFSET(ratio),   AV_OPT_TYPE_FLOAT, { .dbl = 0.996 }, 0, 1, FLAGS },
    { "type", "",OFFSET(type),   AV_OPT_TYPE_INT, { .i64 = 0 }, 0, 1, FLAGS },
    { "hdr2sdr", "",OFFSET(hdr2sdr),   AV_OPT_TYPE_INT, { .i64 = 0 }, 0, 1, FLAGS },
    { NULL }

};// TODO: add something if needed

static const AVClass tenfilter_class = {
    .class_name       = "tenfilter",
    .item_name        = av_default_item_name,
    .option           = tvefilter_options,
    .version          = LIBAVUTIL_VERSION_INT,
    .category         = AV_CLASS_CATEGORY_FILTER,
};

static const AVFilterPad avfilter_vf_tenfilter_inputs[] = {
    {
        .name         = "tenfilter_inputpad",
        .type         = AVMEDIA_TYPE_VIDEO,
        .filter_frame = filter_frame,
        .config_props = config_input,
    },
#ifndef FFMPEG5_SDK_VERSION
    { NULL }
#endif
};

static const AVFilterPad avfilter_vf_tenfilter_outputs[] = {
    {
        .name = "tenfilter_outputpad",
        .type = AVMEDIA_TYPE_VIDEO,
        .request_frame = request_frame,
        .config_props = config_output,
    },
#ifndef FFMPEG5_SDK_VERSION
    { NULL }
#endif
};

#ifndef FFMPEG5_SDK_VERSION
AVFilter ff_vf_tenfilter = {
#else
const AVFilter ff_vf_tenfilter = {
#endif
    .name           = "tenfilter",
    .description    = NULL_IF_CONFIG_SMALL("enhance of video"),
    .priv_size      = sizeof(TveFilterContext),
    .priv_class     = &tenfilter_class,
    .init          = init,
    .uninit        = uninit,
#ifdef FFMPEG5_SDK_VERSION
    FILTER_PIXFMTS_ARRAY(pix_fmts),
    FILTER_INPUTS(avfilter_vf_tenfilter_inputs),
    FILTER_OUTPUTS(avfilter_vf_tenfilter_outputs),
#else
    .query_formats = query_formats,
    .inputs         = avfilter_vf_tenfilter_inputs,
    .outputs        = avfilter_vf_tenfilter_outputs,
#endif
	.flags         = AVFILTER_FLAG_SLICE_THREADS,
};
