/*
+ * libtenav1 encoder
+ *
+ * Copyright (c) 2018-2021 Tencent
+ *
+ * This file is part of FFmpeg.
+ *
+ * FFmpeg is free software; you can redistribute it and/or
+ * modify it under the terms of the GNU Lesser General Public
+ * License as published by the Free Software Foundation; either
+ * version 2.1 of the License, or (at your option) any later version.
+ *
+ * FFmpeg is distributed in the hope that it will be useful,
+ * but WITHOUT ANY WARRANTY; without even the implied warranty of
+ * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
+ * Lesser General Public License for more details.
+ *
+ * You should have received a copy of the GNU Lesser General Public
+ * License along with FFmpeg; if not, write to the Free Software
+ * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
+ */

#include "libtscsdk.h"

#include "libavutil/internal.h"
#include "libavutil/common.h"
#include "libavutil/opt.h"
#include "libavutil/pixdesc.h"
#include "avcodec.h"
#include "internal.h"
#include "dlfcn.h"
#include "string.h"
#include "memory.h"

#ifdef FFMPEG5_SDK_VERSION
#include "encode.h"
#endif

typedef struct tenav1Context
{
    const AVClass *class;
    AVBSFContext *bsf;
    tenav1_t        *encoder;
    tenav1_param_t   params;
    tenav1_output_t  outpic;
    tenav1_inpic_t   in_pic;

    float crf;
    int   cqp;
    int   preset;
    int   bitdepth;

    // ffmpeg param
    char *tenav1_opts;
    char *sdk_config;
    char *ext_para;
} tenav1Context;

static tenav1_inpic_t *inpic_alloc(tenav1_inpic_t *inpic, unsigned int width, unsigned int height) {
  unsigned int align = 32;
  unsigned int stride_in_bytes = (width + 2 * 0 + align - 1) & ~(align - 1);
  memset(inpic, 0, sizeof(tenav1_inpic_t));
  inpic->width = width;
  inpic->height = height;
  inpic->stride[TEN_AV1_PLANE_Y] = stride_in_bytes;
  inpic->stride[TEN_AV1_PLANE_U] = inpic->stride[TEN_AV1_PLANE_V] = stride_in_bytes >> 1;
  inpic->planes[TEN_AV1_PLANE_Y] = NULL;
  inpic->planes[TEN_AV1_PLANE_U] = NULL;
  inpic->planes[TEN_AV1_PLANE_V] = NULL;
  inpic->i_pts = 0;

  return inpic;
}

static void outpic_alloc(tenav1_output_t *outpic) {
  outpic->i_key_frame = 0;
  outpic->i_total_size = 0;
  outpic->i_pts = 0;
  outpic->i_dts = -1;
  outpic->i_nal = 0;
  for (int32_t nal = 0; nal < TEN_MAX_OBU_NUM; nal++) {
    outpic->nal[nal].i_payload = 0;
    outpic->nal[nal].payload = NULL;
  }
}

static av_cold int tenav1_close(AVCodecContext *avctx)
{
    tenav1Context *ctx = avctx->priv_data;
    if (ctx->encoder)
        tenav1_encoder_close(ctx->encoder, &ctx->ext_para);

    av_bsf_free(&ctx->bsf);
    av_log(avctx, AV_LOG_INFO, "tenav1_close\n");

    return 0;
}
static av_cold int tenav1_init(AVCodecContext *avctx)
{
    tenav1Context *ctx = avctx->priv_data;
    ctx->bitdepth = av_pix_fmt_desc_get(avctx->pix_fmt)->comp[0].depth;
    av_log(avctx, AV_LOG_INFO,"bit depth %d\n", av_pix_fmt_desc_get(avctx->pix_fmt)->comp[0].depth);

    av_log(avctx, AV_LOG_INFO, "Tencent AV1 Version: [%s]\n", tenav1_version());
    tenav1_param_default(&ctx->params);
    if (ctx->preset >= 0) {
        ctx->params.preset = ctx->preset;
        if (tenav1_param_config_preset(&ctx->params) < 0) {
            av_log(avctx, AV_LOG_ERROR, "Error setting preset %d, preset should between[0,10]/.\n", ctx->preset);
            return AVERROR(EINVAL);
        }
    }

    ctx->params.pool_threads = avctx->thread_count;
    ctx->params.width = avctx->width;
    ctx->params.height = avctx->height;
    ctx->params.timing_info_type = 1;
    ctx->params.color_primaries = avctx->color_primaries;
    ctx->params.matrix_coefficients = avctx->colorspace;
    ctx->params.transfer_characteristics = avctx->color_trc;
    ctx->params.color_range = (avctx->color_range == AVCOL_RANGE_JPEG ? 1 : 0);

    if (ctx->crf >= 0) {
        ctx->params.crf = ctx->crf;
        ctx->params.rc_mode = 1;  // crf
    } else if (ctx->cqp >= 0) {
        ctx->params.qp = ctx->cqp;
        ctx->params.rc_mode = 2; // cqp
    } else if (avctx->bit_rate > 0) {
        ctx->params.target_bitrate = avctx->bit_rate / 1000;
        ctx->params.rc_mode = 0; // abr
    }

    {
        ctx->params.timebase.den = avctx->time_base.den;
        ctx->params.timebase.num = avctx->time_base.num;
    }
    if (avctx->framerate.num > 0 && avctx->framerate.den > 0) {
        ctx->params.fps.num = avctx->framerate.num;
        ctx->params.fps.den = avctx->framerate.den;
    } else {
        ctx->params.fps.num = avctx->time_base.den;
        ctx->params.fps.den = avctx->time_base.num * avctx->ticks_per_frame;
    }

    av_log(avctx, AV_LOG_DEBUG, "den %d, num %d\n", avctx->time_base.den, avctx->time_base.num);

    if (ctx->tenav1_opts) {
        AVDictionary *     dict = NULL;
        AVDictionaryEntry *en = NULL;

        if (!av_dict_parse_string(&dict, ctx->tenav1_opts, "=", ":", 0)) {
            while ((en = av_dict_get(dict, "", en, AV_DICT_IGNORE_SUFFIX))) {
                int parse_ret = tenav1_param_parse(&ctx->params, en->key, en->value);
                switch (parse_ret) {
                case TEN_AV1_PARAM_BAD_NAME:
                    av_log(avctx, AV_LOG_WARNING, "Unknown option: %s.\n", en->key);
                    break;
                case TEN_AV1_PARAM_BAD_VALUE:
                    av_log(avctx, AV_LOG_WARNING, "Invalid value for %s: %s.\n", en->key, en->value);
                    break;
                default:
                    break;
                }
            }
            av_dict_free(&dict);
        }
    }

    inpic_alloc(&ctx->in_pic, ctx->params.width, ctx->params.height);
    outpic_alloc(&ctx->outpic);
    ctx->encoder = tenav1_encoder_open(&ctx->params, &ctx->ext_para, ctx->sdk_config);
    if (!ctx->encoder) {
        av_log(avctx, AV_LOG_ERROR, "Cannot open libtenav1 encoder.\n");
        return AVERROR_INVALIDDATA;
    }

    if (avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER) {
        const AVBitStreamFilter *av_filter = av_bsf_get_by_name("extract_extradata");
        int ret_val;

        if (!av_filter) {
            av_log(avctx, AV_LOG_ERROR, "extract_extradata bitstream filter "
                   "not found. This is a bug, please report it.\n");
            return AVERROR_BUG;
        }
        ret_val = av_bsf_alloc(av_filter, &ctx->bsf);
        if (ret_val < 0)
            return ret_val;

        ret_val = avcodec_parameters_from_context(ctx->bsf->par_in, avctx);
        if (ret_val < 0)
           return ret_val;

        ret_val = av_bsf_init(ctx->bsf);
        if (ret_val < 0)
           return ret_val;
    }

    av_log(NULL, AV_LOG_DEBUG, "tenav1_encode_init success\n");
    return 0;
}

static int tenav1_encode(AVCodecContext *avctx, AVPacket *pkt, const AVFrame *pic, int *got_packet)
{
    tenav1Context *ctx = avctx->priv_data;
    uint8_t *    dst;
    int          payload = 0;
    int          ret;
    int          i;
    // static int   frame_count = 1;
    const int pixel_num = ctx->bitdepth == 8 ? 1 : 2;
   // static int   frame_count = 1;
    av_log(avctx, AV_LOG_DEBUG, "tenav1_encode_frame.\n");
    if (pic) {
        for (i = 0; i < 3; i++) {
            // printf("stride %d %d\n", ctx->in_pic.stride[i], pic->linesize[i]);
            ctx->in_pic.planes[i] = pic->data[i];
            ctx->in_pic.stride[i] = pic->linesize[i] / pixel_num;
        }
        ctx->in_pic.i_pts = pic->pts;
        ctx->in_pic.i_type = pic->pict_type == AV_PICTURE_TYPE_I ? TEN_KEY_FRAME: TEN_AUTO_FRAME;
        av_log(avctx, AV_LOG_DEBUG, "pts:%ld\n", pic->pts);
    }

    av_log(avctx, AV_LOG_DEBUG, "187\n");
    ret = tenav1_encoder_encode(ctx->encoder, pic ? &ctx->in_pic : NULL, &ctx->outpic,
            &ctx->ext_para);
    av_log(avctx, AV_LOG_DEBUG, "189\n");
    av_log(avctx, AV_LOG_DEBUG, "tenav1_encode_encode_after ret;%d.\n", ret);

    // if (!ret)
    //     return 0;

    if (!ctx->outpic.i_nal)
        return 0;

   for (i = 0; i < ctx->outpic.i_nal; i++)
        payload += ctx->outpic.nal[i].i_payload;

   av_log(avctx, AV_LOG_DEBUG, "Get payload %d\n", payload);

#ifdef FFMPEG5_SDK_VERSION
   ret = ff_alloc_packet(avctx, pkt, payload);
#else
   ret = ff_alloc_packet2(avctx, pkt, payload, 0);
#endif
    if (ret < 0) {
        av_log(avctx, AV_LOG_ERROR, "Error getting output packet.\n");
        return ret;
   }

    dst = pkt->data;

    av_log(avctx, AV_LOG_DEBUG, "212 ret %d\n", ret);
    for (i = 0; i < ctx->outpic.i_nal; i++) {
        memcpy(dst, ctx->outpic.nal[i].payload, ctx->outpic.nal[i].i_payload);
       dst += ctx->outpic.nal[i].i_payload;

        if (ctx->outpic.i_key_frame)
            pkt->flags |= AV_PKT_FLAG_KEY;
    }

    pkt->dts = ctx->outpic.i_dts;
    pkt->pts = ctx->outpic.i_pts;
    // pkt->duration = 1;
    pkt->size = ctx->outpic.i_total_size;
    av_log(avctx, AV_LOG_DEBUG, "Output packet pts:%ld, dts:%ld\n", pkt->pts, pkt->dts);
    // if (pkt->dts > pkt->pts) {
    //     av_log(avctx, AV_LOG_ERROR, "ERROR dts > pts: pts:%ld, dts:%ld\n", pkt->pts, pkt->dts);
   // }

    if (avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER) {
        int ret_val = av_bsf_send_packet(ctx->bsf, pkt);
        if (ret_val < 0) {
            av_log(avctx, AV_LOG_ERROR, "extract_extradata filter "
                   "failed to send input packet\n");
            return ret_val;
        }
        ret_val = av_bsf_receive_packet(ctx->bsf, pkt);
        if (ret_val < 0) {
            av_log(avctx, AV_LOG_ERROR, "extract_extradata filter "
                   "failed to receive output packet\n");
            return ret_val;
        }
    }

    *got_packet = 1;

    return 0;
}

#define OFFSET(x) offsetof(tenav1Context, x)
#define VE AV_OPT_FLAG_VIDEO_PARAM | AV_OPT_FLAG_ENCODING_PARAM
static const AVOption tenav1_options[] = {
    {"tenav1-params", "set the AV1 configuration using a :-separated list of key=value parameters", OFFSET(tenav1_opts), AV_OPT_TYPE_STRING, {0}, 0, 0, VE},
    {"preset", "set the AV1 preset", OFFSET(preset), AV_OPT_TYPE_INT, {.i64 = -1}, -1, INT_MAX, VE},
    {"sdk_config", "sdk config path and file, default : ./sdk_config",  OFFSET(sdk_config), AV_OPT_TYPE_STRING, {0}, 0, 0, VE},
    {"qp", "Constant quantization parameter rate control method", OFFSET(cqp), AV_OPT_TYPE_INT, {.i64 = -1}, -1, INT_MAX, VE}, {NULL}};

static const AVClass class = {
    .class_name = "libtenav1",
    .item_name = av_default_item_name,
    .option = tenav1_options,
    .version = LIBAVUTIL_VERSION_INT,
};

static const AVCodecDefault tenav1_defaults[] = {
    {NULL},
};

AVCodec ff_libtenav1_encoder = {
   .name = "libtenav1",
    .long_name = "libtenav1 AV1",
    .type = AVMEDIA_TYPE_VIDEO,
   .id = AV_CODEC_ID_AV1,
    .priv_data_size = sizeof(tenav1Context),
    .init = tenav1_init,
    .encode2 = tenav1_encode,
    .close = tenav1_close,
    .capabilities = AV_CODEC_CAP_DELAY | AV_CODEC_CAP_AUTO_THREADS,
    // .profiles       = NULL_IF_CONFIG_SMALL(ff_tenav1_profiles),
    .priv_class = &class,
    .defaults = tenav1_defaults,
    // .init_static_data = tenav1_init_static,
    .wrapper_name = "libtenav1",
};
