/*
 * HEVC encoding using the t264 library
 * Copyright (C) 2020 bonniehou
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * no alloc input memory
 */
#include <unistd.h>
#include <float.h>
#include "libavutil/internal.h"
#include "libavutil/common.h"
#include "libavutil/opt.h"
#include "libavutil/pixdesc.h"
#include "libavutil/time.h"
#include "avcodec.h"
#include "internal.h"
#include "dlfcn.h"

#include "libtscsdk.h"

#ifdef FFMPEG5_SDK_VERSION
#include "encode.h"
#endif

#define MAX_BS_FACTOR 1.5
#define MASTER_SLAVE_SAME_IDR 0

enum TI265_RC_METHODS
{
    TEN265_RC_CQP = 0,
    TEN265_RC_ABR_VBV = 1,
    TEN265_RC_ABR = 2,
    TEN265_RC_CRF = 3
};

typedef struct ten265Context
{
    const AVClass *class;
    ten265_t        *encoder;
    ten265_param_t  *params;
    ten265_outpic_t *outpic;
    ten265_inpic_t   in_pic;
    ten265_nal_t    *nal;
    int            qOffsetSize;

    int   avcintra_class;
    float crf;
    float vquality;
    int   cqp;
    int   preset;
    int   reconfig;
    int   forced_idr;

    // ffmpeg param
    char *ten265_opts;
    char *sdk_config;
    char *ext_para;
} ten265Context;

static void TEN265_log(void *p, int level, const char *fmt, va_list args)
{
    static const int level_map[] = {
        [TEN265_LOG_ERROR]   = AV_LOG_ERROR,
        [TEN265_LOG_WARNING] = AV_LOG_WARNING,
        [TEN265_LOG_INFO]    = AV_LOG_INFO,
        [TEN265_LOG_DEBUG]   = AV_LOG_DEBUG};

    if (level == -1) {
        level = TEN265_LOG_INFO;
    }

    if (level < 0 || level > TEN265_LOG_DEBUG)
        return;

    av_vlog(p, level_map[level], fmt, args);
}
static int is_keyframe(int naltype)
{
    switch (naltype)
    {
    case TEN_NAL_UNIT_CODED_SLICE_BLA_W_LP:
    case TEN_NAL_UNIT_CODED_SLICE_BLA_W_RADL:
    case TEN_NAL_UNIT_CODED_SLICE_BLA_N_LP:
    case TEN_NAL_UNIT_CODED_SLICE_IDR_W_RADL:
    case TEN_NAL_UNIT_CODED_SLICE_IDR_N_LP:
    case TEN_NAL_UNIT_CODED_SLICE_CRA:
        return 1;
    default:
        return 0;
    }
}

static void reconfig_encoder(AVCodecContext *avctx)
{
    ten265Context *ctx = avctx->priv_data;
    if (ctx->reconfig) {
        int needreconfig = 0;
        int rcmode = ctx->params->m_iRateControlMode;
        if (ctx->params->m_vbvBufferSize != avctx->rc_buffer_size / 1000 ||
            ctx->params->m_vbvMaxBitrate != avctx->rc_max_rate / 1000) {
            needreconfig = 1;
            ctx->params->m_vbvBufferSize = avctx->rc_buffer_size / 1000;
            ctx->params->m_vbvMaxBitrate = avctx->rc_max_rate / 1000;
        }

        if ((rcmode == TEN265_RC_ABR || rcmode == TEN265_RC_ABR_VBV) &&
            ctx->params->m_fRCTargetBitRate != avctx->bit_rate / 1000) {
            needreconfig = 1;
            if (rcmode == TEN265_RC_ABR_VBV)
                ctx->params->m_vbvBufferSize = ctx->params->m_vbvMaxBitrate = 0;
            ctx->params->m_fRCTargetBitRate = avctx->bit_rate / 1000;
        }

        if (rcmode == TEN265_RC_CRF && ctx->params->m_rfConstant != ctx->crf) {
            needreconfig = 1;
            ctx->params->m_rfConstant = ctx->crf;
        }

        if (needreconfig) {
            ten265_encoder_reconfig(ctx->encoder, ctx->params);
            av_log(avctx, AV_LOG_INFO, "reconfig ten265 params\n");
        }
    }
}

static void init_picture_buffer(ten265Context *ctx)
{
    int num_out_pic = 1;
    ctx->outpic = (ten265_outpic_t *)malloc(sizeof(ten265_outpic_t) * num_out_pic);

    for (int i = 0; i < num_out_pic; i++) {
        ctx->outpic[i].bitstream = NULL;
    }
}

static av_cold int libten265_encode_close(AVCodecContext *avctx)
{
    ten265Context *ctx = avctx->priv_data;
    if (ctx->encoder)
        ten265_encoder_close(ctx->encoder, &ctx->ext_para);

    ten265_param_free(ctx->params);
    free(ctx->outpic);
    free(ctx->in_pic.quant_offsets);
    av_log(avctx, AV_LOG_INFO, "ten265_close\n");

    return 0;
}

static av_cold int libten265_encode_init(AVCodecContext *avctx)
{
    int picHeight16x16;
    int picWidth16x16;
    ten265Context *ctx = avctx->priv_data;
    ctx->ext_para = NULL;

    av_log(avctx, AV_LOG_INFO, "Tencent Shannon Encoder(ten265) Version: [%s]\n", ten265_version());
    ctx->params = ten265_param_alloc();
    if (!ctx->params) {
        av_log(avctx, AV_LOG_ERROR, "Could not allocate ten265 param structure.\n");
        return AVERROR(ENOMEM);
    }

    ten265_param_default(ctx->params);
    if (ctx->preset >= 0) {
        ctx->params->m_preset = ctx->preset;
        if (ten265_param_config_preset(ctx->params) < 0) {
            av_log(avctx, AV_LOG_ERROR, "Error setting preset %d, preset should between[0,10]/.\n", ctx->preset);
            return AVERROR(EINVAL);
        }
    }

    ctx->params->pf_log = TEN265_log;
    ctx->params->p_log_private = avctx;
    ctx->params->m_logLevel = TEN265_LOG_INFO;
    ctx->params->m_poolThreads = avctx->thread_count;
    ctx->params->m_iSourceWidth = avctx->width;
    ctx->params->m_iSourceHeight = avctx->height;

    ctx->params->m_iTimeBaseDen = avctx->time_base.den;
    ctx->params->m_iTimeBaseNum = avctx->time_base.num;

    if (avctx->framerate.num > 0 && avctx->framerate.den > 0) {
        ctx->params->m_iFpsNum = avctx->framerate.num;
        ctx->params->m_iFpsDenom = avctx->framerate.den;
    } else {
        ctx->params->m_iFpsNum = avctx->time_base.den;
        ctx->params->m_iFpsDenom = avctx->time_base.num * avctx->ticks_per_frame;
    }

    if ((avctx->color_primaries <= AVCOL_PRI_BT2020 && avctx->color_primaries != AVCOL_PRI_UNSPECIFIED) ||
        (avctx->color_trc <= AVCOL_TRC_BT2020_12 && avctx->color_trc != AVCOL_TRC_UNSPECIFIED) ||
        (avctx->colorspace <= AVCOL_SPC_BT2020_CL && avctx->colorspace != AVCOL_SPC_UNSPECIFIED))
    {

        ctx->params->m_vuiParametersPresentFlag = 1;
        ctx->params->m_videoSignalTypePresentFlag = 1;
        ctx->params->m_colourDescriptionPresentFlag = 1;
        // ten265 validates the parameters internally
        ctx->params->m_colourPrimaries = avctx->color_primaries;
        ctx->params->m_transferCharacteristics = avctx->color_trc;
        ctx->params->m_matrixCoefficients = avctx->colorspace;
    }

    if (avctx->sample_aspect_ratio.num > 0 && avctx->sample_aspect_ratio.den > 0) {
        char sar[12];
        int  sar_num, sar_den;
        av_reduce(&sar_num, &sar_den, avctx->sample_aspect_ratio.num, avctx->sample_aspect_ratio.den, 65535);
        snprintf(sar, sizeof(sar), "%d:%d", sar_num, sar_den);
        if (ten265_param_parse(ctx->params, "sar", sar) == -2) {
            av_log(avctx, AV_LOG_ERROR, "Invalid SAR: %d:%d.\n", sar_num, sar_den);
            return AVERROR_INVALIDDATA;
        }
    }

    if (ctx->crf >= 0) {
        ctx->params->m_rfConstant = ctx->crf;
        ctx->params->m_iRateControlMode = TEN265_RC_CRF;
    } else if (ctx->cqp >= 0) {
        ctx->params->m_iQP = ctx->cqp;
        ctx->params->m_iRateControlMode = TEN265_RC_CQP;
    } else if (avctx->bit_rate > 0)
    {
        ctx->params->m_fRCTargetBitRate = avctx->bit_rate / 1000;
        ctx->params->m_iRateControlMode = TEN265_RC_ABR_VBV;
    }

    if (!(avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER))
        ctx->params->m_bRepeatHeaders = 1;

    if (ctx->ten265_opts) {
        AVDictionary *     dict = NULL;
        AVDictionaryEntry *en = NULL;

        if (!av_dict_parse_string(&dict, ctx->ten265_opts, "=", ":", 0)) {
            while ((en = av_dict_get(dict, "", en, AV_DICT_IGNORE_SUFFIX))) {
                int parse_ret = ten265_param_parse(ctx->params, en->key, en->value);
                switch (parse_ret) {
                case -1:
                    av_log(avctx, AV_LOG_WARNING, "Unknown option: %s.\n", en->key);
                    break;
                case -2:
                    av_log(avctx, AV_LOG_WARNING, "Invalid value for %s: %s.\n", en->key, en->value);
                    break;
                default:
                    break;
                }
            }
            av_dict_free(&dict);
        }
    }

    init_picture_buffer(ctx);
    ctx->encoder = ten265_encoder_open(ctx->params, &ctx->ext_para, ctx->vquality, ctx->sdk_config);
    if (!ctx->encoder) {
        av_log(avctx, AV_LOG_ERROR, "Cannot open libten265 encoder.\n");
        libten265_encode_close(avctx);
        return AVERROR_INVALIDDATA;
    }

    if (avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER) {
        int         nnal;
        ten265_nal_t *nal;
        ten265_encoder_header(ctx->encoder, &nal, &nnal);

        avctx->extradata_size = 0;
        for (int i = 0; i < nnal; i++)
            avctx->extradata_size += nal[i].i_payload;
        if (avctx->extradata_size <= 0) {
            av_log(avctx, AV_LOG_ERROR, "Cannot encode headers.\n");
            libten265_encode_close(avctx);
            return AVERROR_INVALIDDATA;
        }
        avctx->extradata = av_malloc(avctx->extradata_size + AV_INPUT_BUFFER_PADDING_SIZE);
        if (!avctx->extradata) {
            av_log(avctx, AV_LOG_ERROR, "Cannot allocate HEVC header of size %d.\n", avctx->extradata_size);
            libten265_encode_close(avctx);
            return AVERROR(ENOMEM);
        }
        for (int i = 0; i < nnal; i++) {
            memcpy(avctx->extradata, nal[i].payload, nal[i].i_payload);
            avctx->extradata += nal[i].i_payload;
        }
        avctx->extradata -= avctx->extradata_size;
    }

    // just for qpmap
    picHeight16x16 = (ctx->params->m_iSourceHeight + 15) >> 4;
    picWidth16x16 = (ctx->params->m_iSourceWidth + 15) >> 4;
    ctx->qOffsetSize = picHeight16x16 * picWidth16x16;
    ctx->in_pic.quant_offsets = (float *)malloc(ctx->qOffsetSize * sizeof(float));
    memset(ctx->in_pic.quant_offsets, 0, ctx->qOffsetSize * sizeof(float));
    memset(&ctx->in_pic.extra_sei, 0, sizeof(ten265_sei_t));
    ctx->in_pic.opaque = NULL;
    av_log(NULL, AV_LOG_DEBUG, "ten265_encode_init success\n");
    return 0;
}

static int libten265_encode_frame(AVCodecContext *avctx, AVPacket *pkt, const AVFrame *pic, int *got_packet)
{
    ten265Context *ctx = avctx->priv_data;
    uint8_t *    dst;
    int          payload = 0;
    int          ret;
    int          i;
    static int   frame_count = 1;
    av_log(avctx, AV_LOG_DEBUG, "ten265_encode_frame.\n");
    if (pic) {
        for (i = 0; i < 3; i++) {
            ctx->in_pic.plane[i] = pic->data[i];
            ctx->in_pic.stride[i] = pic->linesize[i];
        }

        ctx->in_pic.i_type = pic->pict_type == AV_PICTURE_TYPE_I ?
                             (ctx->forced_idr ? TEN265_TYPE_IDR : TEN265_TYPE_I) :
                             pic->pict_type == AV_PICTURE_TYPE_P ? TEN265_TYPE_P : TEN265_TYPE_AUTO;
        ctx->in_pic.i_pts = pic->pts;
        reconfig_encoder(avctx);
    }

    ret = ten265_encoder_encode(ctx->encoder, pic ? &ctx->in_pic : NULL, ctx->outpic,
            &ctx->ext_para);
    av_log(avctx, AV_LOG_DEBUG, "ten265_encode_encode_after ret;%d.\n", ret);

    if (!ret)
        return 0;

    if (!ctx->outpic->i_nal)
        return 0;

    for (i = 0; i < ctx->outpic->i_nal; i++)
        payload += ctx->outpic->nal[i].i_payload;

#ifdef FFMPEG5_SDK_VERSION
   ret = ff_alloc_packet(avctx, pkt, payload);
#else
   ret = ff_alloc_packet2(avctx, pkt, payload, 0);
#endif

    if (ret < 0) {
        av_log(avctx, AV_LOG_ERROR, "Error getting output packet.\n");
        return ret;
    }

    dst = pkt->data;

    for (i = 0; i < ctx->outpic->i_nal; i++) {
        memcpy(dst, ctx->outpic->nal[i].payload, ctx->outpic->nal[i].i_payload);
        dst += ctx->outpic->nal[i].i_payload;

        if (is_keyframe(ctx->outpic->nal[i].i_type))
            pkt->flags |= AV_PKT_FLAG_KEY;
    }

    pkt->dts = ctx->outpic->i_dts;
    pkt->pts = ctx->outpic->i_pts;
    if (pkt->dts > pkt->pts) {
        av_log(avctx, AV_LOG_ERROR, "ERROR dts > pts: pts:%ld, dts:%ld\n", pkt->pts, pkt->dts);
    }
    *got_packet = 1;

    return 0;
}

#define OFFSET(x) offsetof(ten265Context, x)
#define VE AV_OPT_FLAG_VIDEO_PARAM | AV_OPT_FLAG_ENCODING_PARAM
static const AVOption options[] = {{"ten265-params", "set the ten265 configuration using a :-separated list of key=value parameters", OFFSET(ten265_opts), AV_OPT_TYPE_STRING, {0}, 0, 0, VE},
    {"forced-idr", "if forcing keyframes, force them as IDR frames", OFFSET(forced_idr), AV_OPT_TYPE_BOOL, {.i64 = 0}, 0, 1, VE},
    {"preset", "set the ten265 preset", OFFSET(preset), AV_OPT_TYPE_INT, {.i64 = -1}, -1, INT_MAX, VE},
    {"sdk_config", "sdk config path and file, default : ./sdk_config",  OFFSET(sdk_config), AV_OPT_TYPE_STRING, {0}, 0, 0, VE},
    {"reconfig", "enable reconfig bitrate/vbv/crf params", OFFSET(reconfig), AV_OPT_TYPE_INT, {.i64 = 0}, 0, 1, VE},
    {"crf", "set the ten265 crf", OFFSET(crf), AV_OPT_TYPE_FLOAT, {.dbl = -1}, -1, FLT_MAX, VE},
    {"vquality", "set the vquality ", OFFSET(crf), AV_OPT_TYPE_FLOAT, {.dbl = -1}, -1, FLT_MAX, VE},
    {"qp", "Constant quantization parameter rate control method", OFFSET(cqp), AV_OPT_TYPE_INT, {.i64 = -1}, -1, INT_MAX, VE}, {NULL}};

static const AVClass class = {
    .class_name = "libten265",
    .item_name = av_default_item_name,
    .option = options,
    .version = LIBAVUTIL_VERSION_INT,
};

static const AVCodecDefault ten265_defaults[] = {
    {NULL},
};

AVCodec ff_libten265_encoder = {
    .name = "libten265",
    .long_name = NULL_IF_CONFIG_SMALL("libten265 H.265 / HEVC"),
    .type = AVMEDIA_TYPE_VIDEO,
    .id = AV_CODEC_ID_HEVC,
    .init = libten265_encode_init,
    .encode2 = libten265_encode_frame,
    .close = libten265_encode_close,
    .priv_data_size = sizeof(ten265Context),
    .priv_class = &class,
    .defaults = ten265_defaults,
    .capabilities = AV_CODEC_CAP_DELAY | AV_CODEC_CAP_AUTO_THREADS,
};
