/*
 * H.264 encoding using the ten264 library
 * Copyright (C) 2005  Mans Rullgard <mans@mansr.com>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/eval.h"
#include "libavutil/internal.h"
#include "libavutil/opt.h"
#include "libavutil/mem.h"
#include "libavutil/pixdesc.h"
#include "libavutil/stereo3d.h"
#include "libavutil/intreadwrite.h"
#include "avcodec.h"
#include "internal.h"
#include "libtscsdk.h"

#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define OPT_STR(opt, param)                                                   \
    do {                                                                      \
        int ret;                                                              \
        if ((ret = ten264_param_parse(&x4->params, opt, param)) < 0) { \
            if(ret == TEN264_PARAM_BAD_NAME)                                    \
                av_log(avctx, AV_LOG_ERROR,                                   \
                        "bad option '%s': '%s'\n", opt, param);               \
            else                                                              \
                av_log(avctx, AV_LOG_ERROR,                                   \
                        "bad value for '%s': '%s'\n", opt, param);            \
            return -1;                                                        \
        }                                                                     \
    } while (0)                                                               \

	
typedef struct ten264_picture_opaque {
    AVFrameSideData **side_data;         
    int            nb_side_data;         
} ten264_picture_opaque;

typedef struct TEN264Context {
    AVClass        *class;
    ten264_param_t    params;
    ten264_t         *enc;
    ten264_picture_t  pic;
    uint8_t        *sei;
    int             sei_size;
    char *preset;
    char *profile;
    char *level;
    char *ten264opts;
    float crf;
    float crf_max;
    char *ten264_params;
    int motino;
    int texture;
    int forced_idr;
    int a53_cc;
    int cqp;
    int avcintra_class;
    int scenechange_threshold;
    int psy;
    int weightp;
    int weightb;
    char *psy_rd;
    char *wpredp;
    char *deblock;
    char *partitions;
    char *sdk_config;
    char *extra;
} TEN264Context;

static void TEN264_log(void *p, int level, const char *fmt, va_list args)
{
    static const int level_map[] = {
        [TEN264_LOG_ERROR]   = AV_LOG_ERROR,
        [TEN264_LOG_WARNING] = AV_LOG_WARNING,
        [TEN264_LOG_INFO]    = AV_LOG_INFO,
        [TEN264_LOG_DEBUG]   = AV_LOG_DEBUG
    };

    if (level < 0 || level > TEN264_LOG_DEBUG)
        return;

    av_vlog(p, level_map[level], fmt, args);
}


static int encode_nals(AVCodecContext *ctx, AVPacket *pkt,
                       const ten264_nal_t *nals, int nnal)
{
    TEN264Context *x4 = ctx->priv_data;
    uint8_t *p;
    int i, size = x4->sei_size, ret;

    if (!nnal)
        return 0;

    for (i = 0; i < nnal; i++)
        size += nals[i].i_payload;

    if ((ret = ff_alloc_packet2(ctx, pkt, size, 0)) < 0)
        return ret;

    p = pkt->data;

    /* Write the SEI as part of the first frame. */
    if (x4->sei_size > 0 && nnal > 0) {
        if (x4->sei_size > size) {
            av_log(ctx, AV_LOG_ERROR, "Error: nal buffer is too small\n");
            return -1;
        }
        memcpy(p, x4->sei, x4->sei_size);
        p += x4->sei_size;
        x4->sei_size = 0;
        av_freep(&x4->sei);
    }

    for (i = 0; i < nnal; i++){
        memcpy(p, nals[i].p_payload, nals[i].i_payload);
        p += nals[i].i_payload;
    }

    return 1;
}

static int avfmt2_num_planes(int avfmt)
{
    switch (avfmt) {
    case AV_PIX_FMT_YUV420P:
    case AV_PIX_FMT_YUVJ420P:
    case AV_PIX_FMT_YUV420P9:
    case AV_PIX_FMT_YUV420P10:
    case AV_PIX_FMT_YUV444P:
        return 3;

    case AV_PIX_FMT_BGR0:
    case AV_PIX_FMT_BGR24:
    case AV_PIX_FMT_RGB24:
        return 1;

    default:
        return 3;
    }
}

static int reconfig_encoder(AVCodecContext *ctx, const AVFrame *frame)
{
    TEN264Context *x4 = ctx->priv_data;
    AVFrameSideData *side_data;


  if (x4->avcintra_class < 0) {
    if (x4->params.b_interlaced && x4->params.b_tff != frame->top_field_first) {

        x4->params.b_tff = frame->top_field_first;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }
    if (x4->params.vui.i_sar_height*ctx->sample_aspect_ratio.num != ctx->sample_aspect_ratio.den * x4->params.vui.i_sar_width) {
        x4->params.vui.i_sar_height = ctx->sample_aspect_ratio.den;
        x4->params.vui.i_sar_width  = ctx->sample_aspect_ratio.num;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (x4->params.rc.i_vbv_buffer_size != ctx->rc_buffer_size / 1000 ||
        x4->params.rc.i_vbv_max_bitrate != ctx->rc_max_rate    / 1000) {
        x4->params.rc.i_vbv_buffer_size = ctx->rc_buffer_size / 1000;
        x4->params.rc.i_vbv_max_bitrate = ctx->rc_max_rate    / 1000;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (x4->params.rc.i_rc_method == TEN264_RC_ABR &&
        x4->params.rc.i_bitrate != ctx->bit_rate / 1000) {
        x4->params.rc.i_bitrate = ctx->bit_rate / 1000;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (x4->crf >= 0 &&
        x4->params.rc.i_rc_method == TEN264_RC_CRF &&
        x4->params.rc.f_rf_constant != x4->crf) {
        x4->params.rc.f_rf_constant = x4->crf;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (x4->params.rc.i_rc_method == TEN264_RC_CQP &&
        x4->cqp >= 0 &&
        x4->params.rc.i_qp_constant != x4->cqp) {
        x4->params.rc.i_qp_constant = x4->cqp;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (x4->crf_max >= 0 &&
        x4->params.rc.f_rf_constant_max != x4->crf_max) {
        x4->params.rc.f_rf_constant_max = x4->crf_max;
        ten264_encoder_reconfig(x4->enc, &x4->params);
    }

    if (ctx->flags2 & (1 <<  4))
    {
        AVCodecContext *avctx = ctx;
        if(x4->ten264opts)
        {
            const char *p= x4->ten264opts;
            while(p){
                char param[4096]={0}, val[4096]={0};
                if(sscanf(p, "%4095[^:=]=%4095[^:]", param, val) == 1){
                    OPT_STR(param, "1");
               } else {
                   OPT_STR(param, val);
               }
                p= strchr(p, ':');
                p+=!!p;
            }
        }
        ten264_encoder_reconfig(x4->enc, &x4->params);
        ctx->flags2 &= ~(1 <<  4);
    }
  }

    side_data = av_frame_get_side_data(frame, AV_FRAME_DATA_STEREO3D);
    if (side_data) {
        AVStereo3D *stereo = (AVStereo3D *)side_data->data;
        int fpa_type;

        switch (stereo->type) {
        case AV_STEREO3D_CHECKERBOARD:
            fpa_type = 0;
            break;
        case AV_STEREO3D_COLUMNS:
            fpa_type = 1;
            break;
        case AV_STEREO3D_LINES:
            fpa_type = 2;
            break;
        case AV_STEREO3D_SIDEBYSIDE:
            fpa_type = 3;
            break;
        case AV_STEREO3D_TOPBOTTOM:
            fpa_type = 4;
            break;
        case AV_STEREO3D_FRAMESEQUENCE:
            fpa_type = 5;
            break;
        case AV_STEREO3D_2D:
            fpa_type = 6;
            break;
        default:
            fpa_type = -1;
            break;
        }

        /* Inverted mode is not supported by ten264 */
        if (stereo->flags & AV_STEREO3D_FLAG_INVERT) {
            av_log(ctx, AV_LOG_WARNING,
                   "Ignoring unsupported inverted stereo value %d\n", fpa_type);
            fpa_type = -1;
        }

        if (fpa_type != x4->params.i_frame_packing) {
            x4->params.i_frame_packing = fpa_type;
            ten264_encoder_reconfig(x4->enc, &x4->params);
        }
    }

    // reconfig crf
    side_data = av_frame_get_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
    if (side_data) {
        AVRegionOfInterest *roi = (AVRegionOfInterest *)side_data->data;
        if (roi->top == INT_MIN && roi->bottom == INT_MIN && roi->left == INT_MIN && roi->right == INT_MIN) {
            float new_crf = (roi->qoffset.num * 1.0) / (roi->qoffset.den * 1.0);
            char *buf = av_mallocz(128);
            snprintf(buf, 128, "%f", new_crf);
            av_log(ctx, AV_LOG_WARNING, "Get new crf config %s\n", buf);
            if (ten264_param_parse(&x4->params, "crf", buf) == 0)
                ten264_encoder_reconfig(x4->enc, &x4->params);
            av_free(buf);
        }
    }

    return 0;
}

static AVFrameSideData *ten264_opaque_new_side_data(ten264_picture_opaque* opaque,enum AVFrameSideDataType type,int size)
{                                                                                                                          
    AVFrameSideData *ret, **tmp;                                                                                           

    if (opaque->nb_side_data > INT_MAX / sizeof(*opaque->side_data) - 1)                                                   
        return NULL;                                                                                                       

    tmp = av_realloc(opaque->side_data,                                                                                    
            (opaque->nb_side_data + 1) * sizeof(*opaque->side_data));                                             
    if (!tmp)                                                                                                              
        return NULL;                                                                                                       
    opaque->side_data = tmp;                                                                                               

    ret = av_mallocz(sizeof(*ret));                                                                                        
    if (!ret)                                                                                                              
        return NULL;                                                                                                       

    if (size > 0) {                                                                                                        
        ret->buf = av_buffer_alloc(size);                                                                                  
        if (!ret->buf) {                                                                                                   
            av_freep(&ret);                                                                                                
            return NULL;                                                                                                   
        }                                                                                                                  

        ret->data = ret->buf->data;                                                                                        
        ret->size = size;                                                                                                  
    }                                                                                                                      
    ret->type = type;                                                                                                      

    opaque->side_data[opaque->nb_side_data++] = ret;                                                                       

    return ret;                                                                                                            
}                                                                                                                          

static int TEN264_frame(AVCodecContext *ctx, AVPacket *pkt, const AVFrame *frame,
                      int *got_packet)
{
    TEN264Context *x4 = ctx->priv_data;
    ten264_nal_t *nal;
    int nnal, i, ret;
    ten264_picture_t pic_out = {0};
    int pict_type;

    ten264_picture_init( &x4->pic );
    x4->pic.img.i_csp   = x4->params.i_csp;
    if (x4->params.i_bitdepth > 8)
        x4->pic.img.i_csp |= TEN264_CSP_HIGH_DEPTH;
    x4->pic.img.i_plane = avfmt2_num_planes(ctx->pix_fmt);

    if (frame) {
        for (i = 0; i < x4->pic.img.i_plane; i++) {
            x4->pic.img.plane[i]    = frame->data[i];
            x4->pic.img.i_stride[i] = frame->linesize[i];
        }

        x4->pic.i_pts  = frame->pts;
        
        switch (frame->pict_type) {
        case AV_PICTURE_TYPE_I:
            x4->pic.i_type = x4->forced_idr > 0 ? TEN264_TYPE_IDR
                                                : TEN264_TYPE_KEYFRAME;
            break;
        case AV_PICTURE_TYPE_P:
            x4->pic.i_type = TEN264_TYPE_P;
            break;
        case AV_PICTURE_TYPE_B:
            x4->pic.i_type = TEN264_TYPE_B;
            break;
        default:
            x4->pic.i_type = TEN264_TYPE_AUTO;
            break;
        }
        reconfig_encoder(ctx, frame);

        if (x4->a53_cc) {
            void *sei_data;
            size_t sei_size;

            ret = ff_alloc_a53_sei(frame, 0, &sei_data, &sei_size);
            if (ret < 0) {
                av_log(ctx, AV_LOG_ERROR, "Not enough memory for closed captions, skipping\n");
            } else if (sei_data) {
                x4->pic.extra_sei.payloads = av_mallocz(sizeof(x4->pic.extra_sei.payloads[0]));
                if (x4->pic.extra_sei.payloads == NULL) {
                    av_log(ctx, AV_LOG_ERROR, "Not enough memory for closed captions, skipping\n");
                    av_free(sei_data);
                } else {
                    x4->pic.extra_sei.sei_free = av_free;

                    x4->pic.extra_sei.payloads[0].payload_size = sei_size;
                    x4->pic.extra_sei.payloads[0].payload = sei_data;
                    x4->pic.extra_sei.num_payloads = 1;
                    x4->pic.extra_sei.payloads[0].payload_type = 4;
                }
            }
        }
    }

    do {
        int p;
        if (ten264_encoder_encode(x4->enc, &nal, &nnal, frame? &x4->pic: NULL, &pic_out, &p, 0,0,
                &x4->extra) < 0)
          return AVERROR_EXTERNAL;

        ret = encode_nals(ctx, pkt, nal, nnal);
        if (ret < 0)
            return ret;
    } while (!ret && !frame && ten264_encoder_delayed_frames(x4->enc));

    pkt->pts = pic_out.i_pts;
    pkt->dts = pic_out.i_dts;


    switch (pic_out.i_type) {
    case TEN264_TYPE_IDR:
    case TEN264_TYPE_I:
        pict_type = AV_PICTURE_TYPE_I;
        break;
    case TEN264_TYPE_P:
        pict_type = AV_PICTURE_TYPE_P;
        break;
    case TEN264_TYPE_B:
    case TEN264_TYPE_BREF:
        pict_type = AV_PICTURE_TYPE_B;
        break;
    default:
        pict_type = AV_PICTURE_TYPE_NONE;
    }
    ctx->coded_frame->pict_type = pict_type;

    pkt->flags |= AV_PKT_FLAG_KEY*pic_out.b_keyframe;
    if (ret) {
        ff_side_data_set_encoder_stats(pkt, (pic_out.i_qpplus1 - 1) * FF_QP2LAMBDA, NULL, 0, pict_type);
        ctx->coded_frame->quality = (pic_out.i_qpplus1 - 1) * FF_QP2LAMBDA;
    }

    *got_packet = ret;
    return 0;
}

static av_cold int TEN264_close(AVCodecContext *avctx)
{
    TEN264Context *x4 = avctx->priv_data;

    av_freep(&avctx->extradata);
    av_freep(&x4->sei);

    if (x4->enc) {
        ten264_encoder_close(x4->enc, &x4->extra);
        x4->enc = NULL;
    }

    return 0;
}

static int convert_pix_fmt(enum AVPixelFormat pix_fmt)
{
    switch (pix_fmt) {
    case AV_PIX_FMT_YUV420P:
    case AV_PIX_FMT_YUVJ420P:
    case AV_PIX_FMT_YUV420P9:
    case AV_PIX_FMT_YUV420P10: return TEN264_CSP_I420;
    case AV_PIX_FMT_YUV422P:
    case AV_PIX_FMT_YUVJ422P:
    case AV_PIX_FMT_YUV422P10: return TEN264_CSP_I422;
    case AV_PIX_FMT_YUV444P:
    case AV_PIX_FMT_YUVJ444P:
    case AV_PIX_FMT_YUV444P9:
    case AV_PIX_FMT_YUV444P10: return TEN264_CSP_I444;
    case AV_PIX_FMT_NV12:      return TEN264_CSP_NV12;
    case AV_PIX_FMT_NV16:
    case AV_PIX_FMT_NV20:      return TEN264_CSP_NV16;
#ifdef TEN264_CSP_NV21
    case AV_PIX_FMT_NV21:      return TEN264_CSP_NV21;
#endif
    };
    return 0;
}

#define PARSE_TEN264_OPT(name, var)\
    if (x4->var && ten264_param_parse(&x4->params, name, x4->var) < 0) {\
        av_log(avctx, AV_LOG_ERROR, "Error parsing option '%s' with value '%s'.\n", name, x4->var);\
        return AVERROR(EINVAL);\
    }

static av_cold int TEN264_init(AVCodecContext *avctx)
{
    TEN264Context *x4 = avctx->priv_data;
    AVCPBProperties *cpb_props;
    int sw,sh;

    if (avctx->global_quality > 0)
        av_log(avctx, AV_LOG_WARNING, "-qscale is ignored, -crf is recommended.\n");
    ten264_param_default(&x4->params);

    x4->params.b_deblocking_filter         = avctx->flags & AV_CODEC_FLAG_LOOP_FILTER;

    if (x4->preset)
        if (ten264_param_default_preset(&x4->params, x4->preset) < 0) {
            int i;
            av_log(avctx, AV_LOG_INFO, "Possible presets:");
            for (i = 0; ten264_preset_names[i]; i++)
                av_log(avctx, AV_LOG_INFO, " %s", ten264_preset_names[i]);
            av_log(avctx, AV_LOG_INFO, "\n");
            av_log(avctx, AV_LOG_INFO, "Possible tunes:");
            av_log(avctx, AV_LOG_INFO, "\n");
            return AVERROR(EINVAL);
        }

    if (avctx->level > 0)
        x4->params.i_level_idc = avctx->level;

    x4->params.pf_log               = TEN264_log;
    x4->params.p_log_private        = avctx;
    x4->params.i_log_level          = TEN264_LOG_DEBUG;
    x4->params.i_csp                = convert_pix_fmt(avctx->pix_fmt);
    x4->params.i_bitdepth           = av_pix_fmt_desc_get(avctx->pix_fmt)->comp[0].depth;

    PARSE_TEN264_OPT("weightp", wpredp);

    if (avctx->bit_rate) {
        x4->params.rc.i_bitrate   = avctx->bit_rate / 1000;
        x4->params.rc.i_rc_method = TEN264_RC_ABR;
    }
    x4->params.rc.i_vbv_buffer_size = avctx->rc_buffer_size / 1000;
    x4->params.rc.i_vbv_max_bitrate = avctx->rc_max_rate    / 1000;
    x4->params.rc.b_stat_write      = avctx->flags & AV_CODEC_FLAG_PASS1;

    if (x4->crf >= 0) {
        x4->params.rc.i_rc_method   = TEN264_RC_CRF;
        x4->params.rc.f_rf_constant = x4->crf;
    } else if (x4->cqp >= 0) {
        x4->params.rc.i_rc_method   = TEN264_RC_CQP;
        x4->params.rc.i_qp_constant = x4->cqp;
    }

    if (x4->crf_max >= 0)
        x4->params.rc.f_rf_constant_max = x4->crf_max;

    if (avctx->rc_buffer_size && avctx->rc_initial_buffer_occupancy > 0 &&
        (avctx->rc_initial_buffer_occupancy <= avctx->rc_buffer_size)) {
        x4->params.rc.f_vbv_buffer_init =
            (float)avctx->rc_initial_buffer_occupancy / avctx->rc_buffer_size;
    }

    PARSE_TEN264_OPT("level", level);
    if (x4->scenechange_threshold >= 0)
        x4->params.i_scenecut_threshold = x4->scenechange_threshold;

    else if (x4->level) {
        int i;
        int mbn = AV_CEIL_RSHIFT(avctx->width, 4) * AV_CEIL_RSHIFT(avctx->height, 4);
        int level_id = -1;
        char *tail;
        int scale =1;

        if (!strcmp(x4->level, "1b")) {
            level_id = 9;
        } else if (strlen(x4->level) <= 3){
            level_id = av_strtod(x4->level, &tail) * 10 + 0.5;
            if (*tail)
                level_id = -1;
        }
        if (level_id <= 0)
            av_log(avctx, AV_LOG_WARNING, "Failed to parse level\n");

        for (i = 0; i<ten264_levels[i].level_idc; i++)
            if (ten264_levels[i].level_idc == level_id)
                x4->params.i_frame_reference = av_clip(ten264_levels[i].dpb / mbn / scale, 1, x4->params.i_frame_reference);
    }

    PARSE_TEN264_OPT("psy-rd", psy_rd);
    PARSE_TEN264_OPT("deblock", deblock);
    PARSE_TEN264_OPT("partitions", partitions);

    if (x4->avcintra_class >= 0)
        x4->params.i_avcintra_class = x4->avcintra_class;

    /* Allow specifying the ten264 profile through AVCodecContext. */
    if (!x4->profile)
        switch (avctx->profile) {
        case FF_PROFILE_H264_BASELINE:
            x4->profile = av_strdup("baseline");
            break;
        case FF_PROFILE_H264_HIGH:
            x4->profile = av_strdup("high");
            break;
        case FF_PROFILE_H264_HIGH_10:
            x4->profile = av_strdup("high10");
            break;
        case FF_PROFILE_H264_HIGH_422:
            x4->profile = av_strdup("high422");
            break;
        case FF_PROFILE_H264_HIGH_444:
            x4->profile = av_strdup("high444");
            break;
        case FF_PROFILE_H264_MAIN:
            x4->profile = av_strdup("main");
            break;
        default:
            break;
        }

    if (x4->profile)
        if (ten264_param_apply_profile(&x4->params, x4->profile) < 0) {
            int i;
            av_log(avctx, AV_LOG_ERROR, "Error setting profile %s.\n", x4->profile);
            av_log(avctx, AV_LOG_INFO, "Possible profiles:");
            for (i = 0; ten264_profile_names[i]; i++)
                av_log(avctx, AV_LOG_INFO, " %s", ten264_profile_names[i]);
            av_log(avctx, AV_LOG_INFO, "\n");
            return AVERROR(EINVAL);
        }

    x4->params.i_width          = avctx->width;
    x4->params.i_height         = avctx->height;
    av_reduce(&sw, &sh, avctx->sample_aspect_ratio.num, avctx->sample_aspect_ratio.den, 4096);
    x4->params.vui.i_sar_width  = sw;
    x4->params.vui.i_sar_height = sh;

    x4->params.i_timebase_den = avctx->time_base.den;
    x4->params.i_timebase_num = avctx->time_base.num;
    x4->params.i_fps_num = avctx->time_base.den;
    x4->params.i_fps_den = avctx->time_base.num * avctx->ticks_per_frame;
    x4->params.b_encode_raw_pts = 0;

    x4->params.analyse.b_psnr = avctx->flags & AV_CODEC_FLAG_PSNR;

    x4->params.i_threads      = avctx->thread_count;

    x4->params.b_interlaced   = avctx->flags & AV_CODEC_FLAG_INTERLACED_DCT;

    x4->params.b_open_gop     = 0;

    x4->params.i_slice_count  = avctx->slices;

    x4->params.vui.b_fullrange = avctx->pix_fmt == AV_PIX_FMT_YUVJ420P ||
                                 avctx->pix_fmt == AV_PIX_FMT_YUVJ422P ||
                                 avctx->pix_fmt == AV_PIX_FMT_YUVJ444P ||
                                 avctx->color_range == AVCOL_RANGE_JPEG;

    if (avctx->colorspace != AVCOL_SPC_UNSPECIFIED)
        x4->params.vui.i_colmatrix = avctx->colorspace;
    if (avctx->color_primaries != AVCOL_PRI_UNSPECIFIED)
        x4->params.vui.i_colorprim = avctx->color_primaries;
    if (avctx->color_trc != AVCOL_TRC_UNSPECIFIED)
        x4->params.vui.i_transfer  = avctx->color_trc;

    if (avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER)
        x4->params.b_repeat_headers = 0;

    if(x4->ten264opts){
        const char *p= x4->ten264opts;
        while(p){
            char param[4096]={0}, val[4096]={0};
            if(sscanf(p, "%4095[^:=]=%4095[^:]", param, val) == 1){
                OPT_STR(param, "1");
            }else
                OPT_STR(param, val);
            p= strchr(p, ':');
            p+=!!p;
        }
    }

    if (x4->ten264_params) {
        AVDictionary *dict    = NULL;
        AVDictionaryEntry *en = NULL;

        if (!av_dict_parse_string(&dict, x4->ten264_params, "=", ":", 0)) {
            while ((en = av_dict_get(dict, "", en, AV_DICT_IGNORE_SUFFIX))) {
                if (ten264_param_parse(&x4->params, en->key, en->value) < 0)
                    av_log(avctx, AV_LOG_WARNING,
                           "Error parsing option '%s = %s'.\n",
                            en->key, en->value);
            }

            av_dict_free(&dict);
        }
    }

    // update AVCodecContext with ten264 parameters
    avctx->has_b_frames = x4->params.i_bframe ?
        x4->params.i_bframe_pyramid ? 2 : 1 : 0;
    if (avctx->max_b_frames < 0)
        avctx->max_b_frames = 0;

    avctx->bit_rate = x4->params.rc.i_bitrate*1000;

    x4->enc = ten264_encoder_open(&x4->params, &x4->extra, x4->sdk_config);
    if (!x4->enc)
        return AVERROR_EXTERNAL;

    if (avctx->flags & AV_CODEC_FLAG_GLOBAL_HEADER) {
        ten264_nal_t *nal;
        uint8_t *p;
        int nnal, s, i;

        s = ten264_encoder_headers(x4->enc, &nal, &nnal);
        avctx->extradata = p = av_mallocz(s + AV_INPUT_BUFFER_PADDING_SIZE);
        if (!p)
            return AVERROR(ENOMEM);

        for (i = 0; i < nnal; i++) {
            /* Don't put the SEI in extradata. */
            if (nal[i].i_type == 6) {
                av_log(avctx, AV_LOG_INFO, "%s\n", nal[i].p_payload+25);
                x4->sei_size = nal[i].i_payload;
                x4->sei      = av_malloc(x4->sei_size);
                if (!x4->sei)
                    return AVERROR(ENOMEM);
                memcpy(x4->sei, nal[i].p_payload, nal[i].i_payload);
                continue;
            }
            memcpy(p, nal[i].p_payload, nal[i].i_payload);
            p += nal[i].i_payload;
        }
        avctx->extradata_size = p - avctx->extradata;
    }

    cpb_props = ff_add_cpb_side_data(avctx);
    if (!cpb_props)
        return AVERROR(ENOMEM);
    cpb_props->buffer_size = x4->params.rc.i_vbv_buffer_size * 1000;
    cpb_props->max_bitrate = x4->params.rc.i_vbv_max_bitrate * 1000;
    cpb_props->avg_bitrate = x4->params.rc.i_bitrate         * 1000;

    return 0;
}

static const enum AVPixelFormat pix_fmts_8bit[] = {
    AV_PIX_FMT_YUV420P,
    AV_PIX_FMT_YUVJ420P,
    AV_PIX_FMT_YUV422P,
    AV_PIX_FMT_YUVJ422P,
    AV_PIX_FMT_YUV444P,
    AV_PIX_FMT_YUVJ444P,
    AV_PIX_FMT_NV12,
    AV_PIX_FMT_NV16,
#ifdef TEN264_CSP_NV21
    AV_PIX_FMT_NV21,
#endif
    AV_PIX_FMT_NONE
};
static const enum AVPixelFormat pix_fmts_9bit[] = {
    AV_PIX_FMT_YUV420P9,
    AV_PIX_FMT_YUV444P9,
    AV_PIX_FMT_NONE
};
static const enum AVPixelFormat pix_fmts_10bit[] = {
    AV_PIX_FMT_YUV420P10,
    AV_PIX_FMT_YUV422P10,
    AV_PIX_FMT_YUV444P10,
    AV_PIX_FMT_NV20,
    AV_PIX_FMT_NONE
};
static const enum AVPixelFormat pix_fmts_all[] = {
    AV_PIX_FMT_YUV420P,
    AV_PIX_FMT_YUVJ420P,
    AV_PIX_FMT_YUV422P,
    AV_PIX_FMT_YUVJ422P,
    AV_PIX_FMT_YUV444P,
    AV_PIX_FMT_YUVJ444P,
    AV_PIX_FMT_NV12,
    AV_PIX_FMT_NV16,
    AV_PIX_FMT_NV21,
    AV_PIX_FMT_YUV420P10,
    AV_PIX_FMT_YUV422P10,
    AV_PIX_FMT_YUV444P10,
    AV_PIX_FMT_NV20,
    AV_PIX_FMT_NONE
};

static av_cold void TEN264_init_static(AVCodec *codec)
{
    codec->pix_fmts = pix_fmts_all;
}

#define OFFSET(x) offsetof(TEN264Context, x)
#define VE AV_OPT_FLAG_VIDEO_PARAM | AV_OPT_FLAG_ENCODING_PARAM
static const AVOption options[] = {
    { "preset",        "Set the encoding preset",   OFFSET(preset),        AV_OPT_TYPE_STRING, { .str = "medium" }, 0, 0, VE},
    { "profile",       "Set profile restrictions  ", OFFSET(profile),       AV_OPT_TYPE_STRING, { 0 }, 0, 0, VE},
    {"level", "Specify level ", OFFSET(level), AV_OPT_TYPE_STRING, {.str=NULL}, 0, 0, VE},
    {"ten264opts", "ten264 options", OFFSET(ten264opts), AV_OPT_TYPE_STRING, {.str=NULL}, 0, 0, VE},
    { "crf",           "Select the quality for constant quality mode",    OFFSET(crf),           AV_OPT_TYPE_FLOAT,  {.dbl = -1 }, -1, FLT_MAX, VE },
    { "crf_max",       "In CRF mode, prevents VBV from lowering quality beyond this point.",OFFSET(crf_max), AV_OPT_TYPE_FLOAT, {.dbl = -1 }, -1, FLT_MAX, VE },
    { "qp",            "Constant quantization parameter rate control method",OFFSET(cqp),        AV_OPT_TYPE_INT,    { .i64 = -1 }, -1, INT_MAX, VE },
    { "ten264-params",  "Override the ten264 configuration using a :-separated list of key=value parameters", OFFSET(ten264_params), AV_OPT_TYPE_STRING, { 0 }, 0, 0, VE },
    { "forced-idr",   "If forcing keyframes, force them as IDR frames.",                                  OFFSET(forced_idr),  AV_OPT_TYPE_BOOL,   { .i64 = 0 }, -1, 1, VE },
    {"a53cc",          "Use A53 Closed Captions (if available)",          OFFSET(a53_cc),        AV_OPT_TYPE_BOOL,   {.i64 = 1}, 0, 1, VE},
    { "avcintra-class","AVC-Intra class 50/100/200",                      OFFSET(avcintra_class),AV_OPT_TYPE_INT,     { .i64 = -1 }, -1, 200   , VE},
    { "sc_threshold", "Scene change threshold",                           OFFSET(scenechange_threshold), AV_OPT_TYPE_INT, { .i64 = -1 }, INT_MIN, INT_MAX, VE },
    {"sdk_config", "sdk config path and file, default : ./sdk_config",  OFFSET(sdk_config), AV_OPT_TYPE_STRING, {0}, 0, 0, VE},
    { NULL },
};

static const AVClass ten264_class = {
    .class_name = "libten264",
    .item_name  = av_default_item_name,
    .option     = options,
    .version    = LIBAVUTIL_VERSION_INT,
};

AVCodec ff_libten264_encoder = {
    .name             = "libten264",
    .long_name        = NULL_IF_CONFIG_SMALL("libten264 H.264 / AVC / MPEG-4 AVC / MPEG-4 part 10"),
    .type             = AVMEDIA_TYPE_VIDEO,
    .id               = AV_CODEC_ID_H264,
    .priv_data_size   = sizeof(TEN264Context),
    .init             = TEN264_init,
    .encode2          = TEN264_frame,
    .close            = TEN264_close,
    .capabilities     = AV_CODEC_CAP_DELAY | AV_CODEC_CAP_AUTO_THREADS,
    .priv_class       = &ten264_class,
    .init_static_data = TEN264_init_static,
    .caps_internal    = FF_CODEC_CAP_INIT_THREADSAFE |
                        FF_CODEC_CAP_INIT_CLEANUP,
    .wrapper_name     = "libten264",
};

