
/*****************************************************************************
 *Copyright(C) 2017 - 2021 Tencent.
 *All Rights Reserved.
 ****************************************************************************/

#ifndef SOURCE_LIB_INC_TEN265_H_
#define SOURCE_LIB_INC_TEN265_H_

#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>

/************************************************************************************************************************************************/
//for ten tenav1
#ifdef __cplusplus
extern "C" {
#endif

#define TEN_AV1_LOG_NONE (-1)
#define TEN_AV1_LOG_ERROR 0
#define TEN_AV1_LOG_WARNING 1
#define TEN_AV1_LOG_INFO 2
#define TEN_AV1_LOG_DEBUG 3

// RC
#define PRINT_STATS 0

#define TEN_AV1_FOURCC 0x31305641

/* error code */
#define RET_ERROR_PARAM 1 /* param error */

/* Image data pointers. */
#define TEN_AV1_PLANE_Y 0 /**< Y (Luminance) plane */
#define TEN_AV1_PLANE_U 1 /**< U (Chroma) plane */
#define TEN_AV1_PLANE_V 2 /**< V (Chroma) plane */

/* preset */
#define TEN_BEST -1      /*150: 1*/
#define TEN_PLACEBO 0    /*100 : 1*/
#define TEN_VERYSLOW 1   /*50 : 1*/
#define TEN_SLOWER 2     /*30 : 1*/
#define TEN_TEN_SLOW 3       /*20 : 1*/
#define TEN_UNIVERSAL 4  /* 15 : 1*/
#define TEN_MEDIUM 5     /* 12 : 1*/
#define TEN_FAST 6       /* 10 : 1*/
#define TEN_FASTER 7     /* 10 : 1*/
#define vVERYFAST 8   /* 8 : 1*/
#define TEN_SUPERFAST 9  /* 5 : 1*/
#define TEN_ULTRAFAST 10 /* 5 : 1*/
#define TEN_HYPERFAST 11 /* 2 : 1*/

/* Range of QMS is between first and last value, with offset applied to inter blocks*/
#define TEN_DEFAULT_QM_Y 10
#define TEN_DEFAULT_QM_U 11
#define TEN_DEFAULT_QM_V 12
#define TEN_DEFAULT_QM_FIRST 5
#define TEN_DEFAULT_QM_LAST 9
#define TEN_MAX_MESH_STEP 4

#define TEN_MAX_OBU_NUM 32 /* maximum number of nal in tenav1_outpic_t */

#define TEN_MINQ 0
#define TEN_MAXQ 255
typedef enum tenav1_mode_t {
  // Good Quality Fast Encoding. The encoder balances quality with the amount of
  // time it takes to encode the output. Speed setting controls how fast.
    TEN_GOOD,
  // Realtime Fast Encoding. Will force some restrictions on bitrate constraints.
    TEN_REALTIME
} tenav1_mode_t;

typedef enum ten_timing_info { TEN_AV1_TIMING_UNSPECIFIED = 0, TEN_AV1_TIMING_EQUAL = 1, TEN_AV1_TIMING_DEC_MODEL = 2 } tenav1_timing_info;
typedef enum tenav1_aq_mode_t {
    TEN_NO_AQ = 0,
    TEN_VARIANCE_AQ = 1,
    TEN_COMPLEXITY_AQ = 2,
    TEN_CYCLIC_REFRESH_AQ = 3,
    TEN_AQ_MODE_COUNT  // This should always be the last member of the enum
} tenav1_aq_mode_t;

typedef enum tenav1_delta_q_mode_t {
  TEN_AV1_AQ_NONE = 0,
  TEN_AV1_AQ_VARIANCE = 1,
  TEN_AV1_AQ_AUTO_VARIANCE = 2,
  TEN_AV1_AQ_AUTO_VARIANCE_BIASED = 3,
  TEN_AV1_AQ_VARIANCE_OPT = 4,  // use tenav1 rate model to calculate Qindex offset.
  TEN_AV1_AQ_PERCEPTUAL = 5,    // Modulation to improve perceptual quality
  TEN_AV1_AQ_NUM                // This should always be the last member of the enum
} tenav1_delta_q_mode_t;

/*!\brief List of supported color primaries */
typedef enum tenav1_color_primaries_t {
  TEN_AV1_CICP_CP_RESERVED_0 = 0,   /**< For future use */
  TEN_AV1_CICP_CP_BT_709 = 1,       /**< BT.709 */
  TEN_AV1_CICP_CP_UNSPECIFIED = 2,  /**< Unspecified */
  TEN_AV1_CICP_CP_RESERVED_3 = 3,   /**< For future use */
  TEN_AV1_CICP_CP_BT_470_M = 4,     /**< BT.470 System M (historical) */
  TEN_AV1_CICP_CP_BT_470_B_G = 5,   /**< BT.470 System B, G (historical) */
  TEN_AV1_CICP_CP_BT_601 = 6,       /**< BT.601 */
  TEN_AV1_CICP_CP_SMPTE_240 = 7,    /**< SMPTE 240 */
  TEN_AV1_CICP_CP_GENERIC_FILM = 8, /**< Generic film (color filters using illuminant C) */
  TEN_AV1_CICP_CP_BT_2020 = 9,      /**< BT.2020, BT.2100 */
  TEN_AV1_CICP_CP_XYZ = 10,         /**< SMPTE 428 (CIE 1921 XYZ) */
  TEN_AV1_CICP_CP_SMPTE_431 = 11,   /**< SMPTE RP 431-2 */
  TEN_AV1_CICP_CP_SMPTE_432 = 12,   /**< SMPTE EG 432-1  */
  TEN_AV1_CICP_CP_RESERVED_13 = 13, /**< For future use (values 13 - 21)  */
  TEN_AV1_CICP_CP_EBU_3213 = 22,    /**< EBU Tech. 3213-E  */
  TEN_AV1_CICP_CP_RESERVED_23 = 23  /**< For future use (values 23 - 255)  */
} tenav1_color_primaries_t;        /**< alias for enum tenav1_color_primaries */

/*!\brief List of supported transfer functions */
typedef enum tenav1_transfer_characteristics_t {
  TEN_AV1_CICP_TC_RESERVED_0 = 0,      /**< For future use */
  TEN_AV1_CICP_TC_BT_709 = 1,          /**< BT.709 */
  TEN_AV1_CICP_TC_UNSPECIFIED = 2,     /**< Unspecified */
  TEN_AV1_CICP_TC_RESERVED_3 = 3,      /**< For future use */
  TEN_AV1_CICP_TC_BT_470_M = 4,        /**< BT.470 System M (historical)  */
  TEN_AV1_CICP_TC_BT_470_B_G = 5,      /**< BT.470 System B, G (historical) */
  TEN_AV1_CICP_TC_BT_601 = 6,          /**< BT.601 */
  TEN_AV1_CICP_TC_SMPTE_240 = 7,       /**< SMPTE 240 M */
  TEN_AV1_CICP_TC_LINEAR = 8,          /**< Linear */
  TEN_AV1_CICP_TC_LOG_100 = 9,         /**< Logarithmic (100 : 1 range) */
  TEN_AV1_CICP_TC_LOG_100_SQRT10 = 10, /**< Logarithmic (100 * Sqrt(10) : 1 range) */
  TEN_AV1_CICP_TC_IEC_61966 = 11,      /**< IEC 61966-2-4 */
  TEN_AV1_CICP_TC_BT_1361 = 12,        /**< BT.1361 */
  TEN_AV1_CICP_TC_SRGB = 13,           /**< sRGB or sYCC*/
  TEN_AV1_CICP_TC_BT_2020_10_BIT = 14, /**< BT.2020 10-bit systems */
  TEN_AV1_CICP_TC_BT_2020_12_BIT = 15, /**< BT.2020 12-bit systems */
  TEN_AV1_CICP_TC_SMPTE_2084 = 16,     /**< SMPTE ST 2084, ITU BT.2100 PQ */
  TEN_AV1_CICP_TC_SMPTE_428 = 17,      /**< SMPTE ST 428 */
  TEN_AV1_CICP_TC_HLG = 18,            /**< BT.2100 HLG, ARIB STD-B67 */
  TEN_AV1_CICP_TC_RESERVED_19 = 19     /**< For future use (values 19-255) */
} tenav1_transfer_characteristics_t;  /**< alias for enum tenav1_transfer_function */

/*!\brief List of supported matrix coefficients */
typedef enum tenav1_matrix_coefficients_t {
  TEN_AV1_CICP_MC_IDENTITY = 0,     /**< Identity matrix */
  TEN_AV1_CICP_MC_BT_709 = 1,       /**< BT.709 */
  TEN_AV1_CICP_MC_UNSPECIFIED = 2,  /**< Unspecified */
  TEN_AV1_CICP_MC_RESERVED_3 = 3,   /**< For future use */
  TEN_AV1_CICP_MC_FCC = 4,          /**< US FCC 73.628 */
  TEN_AV1_CICP_MC_BT_470_B_G = 5,   /**< BT.470 System B, G (historical) */
  TEN_AV1_CICP_MC_BT_601 = 6,       /**< BT.601 */
  TEN_AV1_CICP_MC_SMPTE_240 = 7,    /**< SMPTE 240 M */
  TEN_AV1_CICP_MC_SMPTE_YCGCO = 8,  /**< YCgCo */
  TEN_AV1_CICP_MC_BT_2020_NCL = 9,  /**< BT.2020 non-constant luminance, BT.2100 YCbCr  */
  TEN_AV1_CICP_MC_BT_2020_CL = 10,  /**< BT.2020 constant luminance */
  TEN_AV1_CICP_MC_SMPTE_2085 = 11,  /**< SMPTE ST 2085 YDzDx */
  TEN_AV1_CICP_MC_CHROMAT_NCL = 12, /**< Chromaticity-derived non-constant luminance */
  TEN_AV1_CICP_MC_CHROMAT_CL = 13,  /**< Chromaticity-derived constant luminance */
  TEN_AV1_CICP_MC_ICTCP = 14,       /**< BT.2100 ICtCp */
  TEN_AV1_CICP_MC_RESERVED_15 = 15  /**< For future use (values 15-255)  */
} tenav1_matrix_coefficients_t;

/*!\brief List of chroma sample positions */
typedef enum tenav1_chroma_sample_position_t {
  TEN_AV1_CSP_UNKNOWN = 0,          /**< Unknown */
  TEN_AV1_CSP_VERTICAL = 1,         /**< Horizontally co-located with luma(0, 0)*/
                                /**< sample, between two vertical samples */
  TEN_AV1_CSP_COLOCATED = 2,        /**< Co-located with luma(0, 0) sample */
  TEN_AV1_CSP_RESERVED = 3          /**< Reserved value */
} tenav1_chroma_sample_position_t; /**< alias for enum tenav1_transfer_function */

/*!\brief OBU types. */
typedef enum TenObuType {
    TEN_OBU_SEQUENCE_HEADER = 1,
    TEN_OBU_TEMPORAL_DELIMITER = 2,
    TEN_OBU_FRAME_HEADER = 3,
    TEN_OBU_TILE_GROUP = 4,
    TEN_OBU_METADATA = 5,
    TEN_OBU_FRAME = 6,
    TEN_OBU_REDUNDANT_FRAME_HEADER = 7,
    TEN_OBU_TILE_LIST = 8,
    TEN_OBU_VTEN_AV1_COPY_RIGHT = 9,  // reserve obu.
    TEN_OBU_PADDING = 15,
    TEN_OBU_COUNT
} TenObuType;

/*!\brief Rate control mode */
typedef enum tenav1_rc_mode {
  TEN_AV1_ABR, /**< Average Bit Rate (ABR) mode */
  TEN_AV1_CRF, /**< Constant Rate Factor (CRF) mode */
  TEN_AV1_Q,   /**< Constant Quality (Q) mode */
  TEN_AOM_VBR
} tenav1_rc_mode;

typedef enum tenav1_color_format_t {
    TEN_k400 = 0, /* Y */
    TEN_k420 = 1, /* YUV420 */
    TEN_k422 = 2, /* YUV422 */
    TEN_k444 = 3  /* YUV444 */
} tenav1_color_format_t;

enum TenFrameType {
    TEN_AUTO_FRAME = -1,
    TEN_KEY_FRAME = 0,
    TEN_INTER_FRAME = 1,
    TEN_INTRA_ONLY_FRAME = 2,  // replaces intra-only
    TEN_S_FRAME = 3,
    TEN_FRAME_TYPES,
};

/**\brief Image Descriptor */
typedef struct tenav1_inpic_t {
  unsigned int width;  /**< image width */
  unsigned int height; /**< image height */
  void *planes[3];     /**< pointer to the top left pixel for each plane */
  int stride[3];       /**< stride between rows for each plane */
  int64_t i_pts;       /**< input pts */
  int32_t i_type;      /**< input frame type */
  int *quant_offsets;  // quant_offsets of the 16x16 blocks in the pic
} tenav1_inpic_t;

typedef struct tenav1_timing_info_t {
  uint32_t num_units_in_display_tick;
  uint32_t time_scale;
  int equal_picture_interval;
  uint32_t num_ticks_per_picture;
} tenav1_timing_info_t;

typedef struct tenav1_dec_model_info_t {
  uint32_t num_units_in_decoding_tick;
  int encoder_decoder_buffer_delay_length;
  int buffer_removal_time_length;
  int frame_presentation_time_length;
} tenav1_dec_model_info_t;

typedef struct TEN_MESH_PATTERN {
  int range;
  int interval;
} TEN_MESH_PATTERN;

/*!\brief Encoder output packet
 *
 * This structure contains the output data the encoder produce while compressing a frame.
 */
typedef struct tenav1_packet_t {
  uint8_t i_key_frame;
  int64_t i_pts;
  int32_t i_type;
  uint8_t *payload;   /**< compressed data buffer */
  uint64_t i_payload; /**< length of compressed data */
} tenav1_packet_t;

typedef struct tenav1_output_t {
  uint8_t i_key_frame;
  int64_t i_total_size;
  int64_t i_pts;
  int64_t i_dts;
  int32_t i_nal;
  double psnr[3];
  double ssim[3];
  tenav1_packet_t nal[TEN_MAX_OBU_NUM];
} tenav1_output_t;

/*!\brief Rational Number
 *
 * This structure holds a fractional value.
 */
typedef struct tenav1_rational_t {
  int num;        /**< fraction numerator */
  int den;        /**< fraction denominator */
} tenav1_rational_t; /**< alias for struct tenav1_rational */

/*!\brief Encoder configuration structure
 *
 * This structure contains the encoder settings that have common representations
 * across all codecs. This doesn't imply that all codecs support all features,
 * however.
 */
typedef struct tenav1_param_t {
  // file
  char *input;
  char *bitstream;
  char *recon_file;
  char *twopass_path;
  int stat_keep;

  // high encode feature
  char tune[20];
  int preset;
  int profile;
  int level_idx_0;  // TODO: remove
  int write_webm_muxer;
  int write_ivf_muxer;
  tenav1_rational_t fps;
  tenav1_rational_t timebase;  // stream timebase units
  int force_vfr_test;
  int enable_monochrome;  // yuv400
  int force_consistency;
  int bit_depth;           // input yuv bitdepth
  int internal_bit_depth;  // internal bit depth for encoding
  int read_buf_cnt;

  // log info
  int log_psnr;
  int log_ssim;
  int log_level;
  void (*pf_log)(void *, int i_level, const char *psz, va_list);
  void *p_log_private;

  // size and position
  int width;               // width of the frame
  int height;              // height of the frame
  int frames_to_encoded;   // max number of frames to encode
  int drop_frames;         // skip the first n frames
  int enable_super_block;  // 1:enable 128x128 CTU; 2: condition from width/height; 3: further condition from frame-thread
  int min_partition_size;  // min partition size [4,8,16,32,64,128]
  int max_partition_size;  // max partition size [4,8,16,32,64,128]
  int uniform_spacing;     // devide tile in uniform way.
  int tile_rows;
  int tile_cols;

  // sei
  int write_copyright;

  // lookahead
  int lag_in_frames;
  int lookahead_delay;
  int enable_lookahead_delay;
  int lookahead_block_size;    // TODO: remove
  int lookahead_pool_threads;  // TODO: remove
  int lookahead_slices;        // only default
  int lookahead_parallel;
  int lookahead_hmvp_thresh;    // only default
  int lookahead_search_method;  // only default
  int lookahead_satd_scale;
  int lookahead_satd_scale_mvp;
  int lookahead_satd_scale_thres1;
  int lookahead_satd_scale_thres2;
  int lookahead_skip_fracme;
  int lookahead_search_range_shift;  // only default
  int lookahead_early_skip;          // only default
  int lookahead_early_skip_intra;
  int lookahead_early_skip_intra1;
  int lookahead_early_skip_intra2;
  int vfr_input;
  int align_input_type;  // align with source picture type, 0: closed, 1 : align with I frame, 2 : align with I / P frame default

  // aq_mode
  int aq_mode;
  double aq_strength;   // only default
  double aq_weightmax;  // TODO: enc_cu
  double aq_weightmin;  // TODO: enc_cu

  // cu_tree
  int cutree;
  int dramatic_scene_cutree_reset;
  double cutree_strength;

  // tpl model
  int enable_tpl_model;
  int tpl_me_method;
  int tpl_length;
  double tpl_length_thres;
  double quant_ratio_strength;
  double tpl_scaling_factor;
  int adapt_weight_in_tpl;
  int tpl_adapt_weight_default;
  double tpl_adaptweight_coeff1;
  double tpl_adaptweight_coeff2;
  double tpl_kf_strength;
  int tpl_lowres_weight;
  int tpl_base_size;
  int tpl_me_size;
  int tpl_4k_me_size;
  int tpl_scale_thres;
  int align_rps_in_tpl;
  int tpl_similar_mvp_thr;
  int tpl_max_mvp_num;
  int tpl_prune_mvp_thr;
  int tpl_prune_ref_mvp_thr;
  int tpl_reorder_ref_frames;
  int tpl_cplx_block_thr;
  int tpl_sse_shift;
  double tpl_deltaq_strength;
  int tpl_interp_taps;
  int tpl_adapt_length;
  int tpl_fast_intra;
  int tpl_intra_edge_filter;
  int tpl_nstep_delta;
  int tpl_cost_sad;

  // arnr
  int arnr_max_frames;
  int arnr_interp_taps;  // TODO: function
  int arnr_me_method;
  int arnr_prune_mvp_thr;
  int arnr_similar_mvp_thr;
  int arnr_prune_16x16_thr;
  int arnr_prune_simple_block;
  int arnr_prune_by_block_color;
  int arnr_strength;
  double arnr_q_decay;
  int filter_second_arf;
  int adapt_arnr_frames;
  int similarity_based_arnr;
  int filter_sec_skip_thres;
  int arnr_parallel_rows;
  double arnr_ref_frame_size;
  double arnr_decays_param[3];
  double arnr_window_error_ratio;
  double arnr_d_factor_min;
  double arnr_similar_frame_param[2];

  // feature
  int zero_latancy;  // only default
  int high_quality_iframe;
  int adaptive_overlay;

  // scene cut
  int enable_scenecut;
  int scenecut_threshold;
  int scenecut_bias;  // only default
  int scenecut_propagate_thr;
  int scenecut_end_seqs;
  int scenecut_filter;     // only default
  int detect_black_white;  // only default
  int detect_transition_to_still;
  int end_judgement;
  double vbv_end_ratio;

  // screen content detection
  int screen_content_detect_opt;

  // gop
  int gop_structure; /* 0(random-access) 1(low-delay) 2(all-intra) */
  int gop_size;
  int adaptive_gop; /* 0(none), 1(fast), 2(trellis?)*/
  int open_gop;
  int keyint_min;  // keyframe minimum interval
  int keyint;      // keyframe maximum interval
  int round_keyint;
  int key_fps_period;
  int still_picture;           // full_still_picture_hdr
  int full_still_picture_hdr;  // full_still_picture_hdr
  // for adapt gop
  int gpb_large_mv_thresh[4];  // threshold for gop 8, 16, 32
  int large_mv_frac_thresh[4];
  int intra_frac_thresh[4];
  int cost_ratio_thresh[4];
  int adapt_cost_ratio0;
  int adapt_cost_ratio1;
  int adapt_diff_ratio;
  int adapt_simple_ratio;
  int adapt_matched_block_propotion;
  int adapt_cplx_block_propotion;
  int gdb_zero_mv_thresh;

  // rate control
  enum tenav1_rc_mode rc_mode;
  int cq_level;  // constrained quality level
  int enable_roi;
  double roi_strength;
  int qp;
  int crf2qp;  // 1:crf2qp to match 265 bitrate, 2:crf2qp to match all qindex(0:255) by linear model
  int stat_write;
  int stat_read;
  int delta_bigq;
  int use_lambda_tab;
  int qpindex_min;
  int deltaq_mode;
  int use_aom_qp_offset;
  int adapt_pb_boost;
  int overflowed_gf_qplevel;
  double overflowed_gf_gap;
  double overflowed_gf_coeff;
  int max_pb_boost;
  int min_pb_boost;
  int ori_weight;
  int new_weight;
  int pb_boost_coeff;
  double lambda_scale_max[2];
  double lambda_scale_min;
  int cu_level_lambda_strength;
  double vmaf_lambda_coeffs[4];  // 0: u, 1: b, 2: k, 3: o.
  int worst_allowed_q;
  int best_allowed_q;
  int rc_min_quantizer;  // min quantizer (best quality)
  int rc_max_quantizer;  // max quantizer (worst quality)
  uint32_t target_bitrate;
  uint32_t vbv_buffer_size;
  uint32_t vbv_maxrate;
  uint8_t is_abraddvbv;
  double vbv_buffer_size_period;
  uint8_t row_level_rc;
  int8_t b_icompen_window;
  double ratio_pred;
  int vbv_clip;
  int qp_step;
  double rc_q_compress;
  double vbv_pred_weight[3];
  double vbv_pred_min_weight[3];
  double complex_scene_thres;
  double vbv_init;
  double vbv_buffer_end;
  double vbv_end_frame_adjust;
  int enable_const_vbv;
  double crf;
  int input_crf;
  double crf_constant_max;
  double crf_constant_min;
  double qcompress;
  double ipratio;
  double pbratio;
  double qp2qstep_power;
  double qp2qstep_base;
  int mbtree_offset;
  int thread_lock;
  double keep_scene_scale;
  int qpres_qthres;
  double overflow_max;
  double overflow_min;
  int adjust_lamda_use_tpl[2];
  double ctu_lamda_max[2];
  double ctu_lamda_min[2];
  double adjust_lamda_noise_level;
  double curve_adjust_lambda_base_ratio[2];
  int worst_quality_calc_iters;  // worst quality calculation iters for cqp_vbv before encode, 0 is close
  int log_rctrace;
  double limit_abnormal_ssim_lamda_factor[2];
  int adjust_lamda_block_min_num;
  double lamda_factor_same_direct_ratio;
  int diff_adjust_direct_num[3];
  double cplxr_sum_factor[4];
  double cplxr_sum_factor_static;
  double cplxr_cost_ratio_thres;
  double cplxr_dark_ratio_thres;
  int rc_sync_dist;
  int abr_init_length;  // set gop=16 for abr init
  int abr_init_gop;
  double common_dark_ratio_thres;
  int dark_clip_qindex;
  int adaptive_vbv;  // 0: close, 1:only overflow, 2: overflow and underflow for simple, 3: overflow and underflow
  double adaptive_vbv_weight;
  double adaptive_vbv_weight_simple;

  // rd opt for aom rate control mode
  int use_neighbor_qp;
  double arf_boost_factor;
  int gfu_boost_thresh;  // lambda rd opt
  int lambda_offset;     // lambda rd opt
  int adaptive_qplevel;
  int shorten_gpgap_offset;
  double adaptive_qp_darksce_thres;
  int qpoffset_limit;
  int qpoffset_limit_low;
  int adaptqscale_satd_thres[3];
  int adapt_qplevel_postive_th;
  int adapt_qplevel_uvsatd;
  int flat_qplevel_for_arf;
  int last_gop_qp_offset;
  int adaptive_lambda_level;
  int enable_1to4_lamda;
  int adaptive_vmaf_size;
  int adaptive_vmaf_interintra;
  int tune_ssim_lambda;
  int tune_vmaf_lambda;
  double lamda_b32_max;
  double tpl_ctu_lamda_max;
  double tpl_ctu_lamda_min;
  double tpl_lamda_pow;
  double tpl_factor_large_diff_min;
  double tpl_factor_large_diff_max;
  double tpl_ctu_factor_diff_max;
  int qp_float_compensation;
  int tpl_ctu_offset_base_frame_layer;
  double tpl_qindex_offset_pow[7];
  int ctu_qindex_max_diff;
  double ctu_lamda_qindex_weight;
  double ctu_lamda_qindex_weight2;
  double ctu_lamda_qindex_weight3;
  // rd opt for v265 rate control mode
  int adaptive_qscale;
  int first_i_inter_thres;
  double first_i_intra_thres;
  int first_i_clip0;
  int adaptive_ipratio;
  int adaptive_pbratio;
  double ratio_thres;
  double min_qscale;
  double max_qscale;
  double max_pbratio;
  double gpb_scale_ratio_thres_min;
  int gpb_scale_inter_thres_min;
  int gpb_scale_inter_thres_max;
  int gpb_scale_intra_thres;
  double gpb_scale_ratio;
  int adaptive_complexity;
  int use_aom_gopqp;

  // filter (deblock cdef restoration)
  int enable_ctu_filter;
  int enable_ctu_filter_layers;
  int filter_param_delay;
  int faster_deblock_search;
  int enable_deblock;
  int enable_cdef;
  int enable_restoration;
  int deblock_pick_method;
  int deblock_pick_dual;  // 0 : only separate. 1 : only dual. 2 : both
  int deblock_step_thres;
  int cdef_pick_method;
  int nb_cdef_strengths_used;
  int uv_dc_delta_q;
  double cdef_lambda_factor;
  int uv_ac_delta_q;
  int cdef_arg1;
  int cdef_arg2;
  int restoration_pick_method;
  int interlace_restoration;
  int enable_lr_uv_shift;
  int enable_lr_maxsize_limit;
  int prune_wiener_based_on_src_var;  // 0 : no pruning 1 : conservative pruning 2 : aggressive pruning
  int enable_sgr_ep_pruning;
  int prune_sgr_based_on_wiener;
  int closed_restoration_type;
  int disable_finerwiener;
  int close_chroma_rest;
  double cdef_sse_factor;
  double rest_sse_factor;
  int delta_q_res_factor;
  int enable_tpl_frame_qp;
  double tpl_frame_qp_i_offset;
  double tpl_frame_qp_p_offset;

  // frame multiple thread
  int frame_threads;
  int fpp_rows_delay;
  int extend_mv_limit;
  int limited_ref_layer;

  // wpp
  int wpp;  // enable row wpp analyse
  int row_ctx_method;
  int pool_threads;

  // tile parallel
  int tile_threads;

  // rdo
  // 0: no ctx update. 1: ctu; 2: cu; 3: tu ( Tu not supported yet.)
  int syntax_skip_layer;    // Specify which frame layer may skip cu level update.
  int syntax_update_freq;   // if syntex update in ctu level, config it for skipping some CTU.
  int syntax_update_level;  // Mode & Coeff cdf and cost update frequency  to enc_cu
  int update_ctu_ctx_freq;
  int mv_update_level;      // Mv cdf and cost update frequency
  int mv_update_range;      // Only update [-mv_update_range, mv_update_range] mv cost (Integer)
                            // When mv_update_level == CTX_UPDATE_CU.

  int rdoq_max_nz_nums;

  tenav1_color_primaries_t color_primaries;                    // TODO: sps
  tenav1_transfer_characteristics_t transfer_characteristics;  // TODO: sps
  tenav1_matrix_coefficients_t matrix_coefficients;            // TODO: sps
  tenav1_chroma_sample_position_t chroma_sample_position;      // TODO: sps ?
  int color_range;                                          // TODO: sps
  int timing_info_present;                                  // TODO: sps
  int timing_info_type;

  // rps
  int bframe_next_layer;
  int long_term;
  int max_reference_frames;    // enable reduced set of references
  int max_reference_frames_b;  // enable reduced set of references of normal B
  int adapt_reference_frames_b;
  int clean_reference_frames;  // reduce reference picture set
  int use_sec_rps;
  int layer_use_prev_gop;

  // to frame
  int allow_intra_layer;
  int interlace_sad_size;
  int interlacex4_sad_size;
  int enable_interlace_sad;
  int enable_me_interlace_sad;
  int using_qm;
  int qm_minlevel;
  int qm_maxlevel;
  int transform_mode; /* transform mode: 0(only 4x4) 1(largest) 2(select) */
  int enable_segment;
  int shorten_gpgap_weight;
  uint8_t cdf_update_mode;
  int enable_order_hint;
  int order_hint_bits;
  int enable_ref_frame_mvs;  // maximum number of references per frame
  int errorthres_skip_tmvp;
  int errorthres_skip_tmvp2;
  int skip_ref_frame_layer;

  int enable_global_motion;       // enable global motion usage for sequence
  int skip_thr_global_motion[3];  // skop global motion alg
  int enable_primary_ref;         // enable primary reference
  int use_lowres_feature_points;  // use lowres to detect feature points
  int enable_warped_motion;       // sequence level
  int enable_screen_content_tools;
  int reduced_tx_type_set;
  int enable_adapt_idtx;

  int reduce_global_motion;
  int hp_mv_qindex[2];
  int prune_ref_frame;
  int prune_ref_frame_for_rect_partitions;

  // in warped_motion
  int gm_inliers_thresh;
  int gm_improve_correspondence;
  int gm_skip_near_identity;
  int prune_gm_refs;
  int prune_gm_refinement;
  int gm_search_type;
  int selective_ref_gm;
  uint8_t gm_erroradv_type;

  // to enc_cu
  int enable_rect_partitions;  // enable rectangular partitions for sequence
  int enable_ab_partitions;    // enable AB partitions for sequence
  int enable_1to4_partitions;  // enable 1:4 and 4:1 partitions for sequence
  int enable_inter_obmc;
  int cu_order_check;  // 4:NONE_OTHER_SPLIT  0:NONE_SPLIT_OTHER 1:SPLIT_NONE_OTHER 2:ADAPT1 3: ADAPT2
  int cu_order_from_var;
  int skip_cur_from_maxsize;
  int neighbor_size_skip_full;
  int cu_topdown_from_tplmv;
  int stop_split_from_nbor_nsplit_cost[6];
  int stop_nrec_from_nbor_nsplit_cost[6];
  int stop_sub_from_nbor_nsplit_cost[6];
  int skip_128_qp_max;
  int skip_sub_from_minsize[6];
  int skip_horver_from_found;
  int skip_rect_partition_depth;
  int stop_sub_from_nbor_nsplit_count;
  int skip_ab4_from_rdcost[6];
  int skip_ab4_from_mixrd[6];
  int skip_ab_from_ml[6];
  int subdiff_skip_split_ratio[6];
  int subdiff_skip_rec_ratio[6];
  int subdiff_skip_split_nocoeff[6];
  int subdiff_skip_split_nocoeff_judge;
  int subdiff_skip_split_nocoeff_factor;
  int subcost_skipfull_thresh[6];
  int costdiff_skip_rec_thres[6];
  int skipver_fromhor_thresh[6];
  int subdiff_skip_ab4_thres[6];
  int subsplit_skip_full_thresh[6];
  int subpart_skip_other_thresh[6];
  int subsplit_skip_rec_thr[6];
  int subnone_skip_rec_thr[6];
  int subsplit_skip_ab4_thresh[6];
  int splitmode_skip_ab4_thres[6];
  int split_skip_hv4_thres[6];
  int subsplit_afterrec_skip_other;
  int skip_ab4_from_mixrd_depth;
  int skip_rect_with_ml[6];
  int prune_inter_from_skipmode[6];
  int prune_inter_from_nonskip[7];
  int prune_split_from_dist[6];
  int prune_split_fromdist_forframe;
  int precost_derive_dir[2];
  int precost_derive_size[6];
  int stop_sub_from_nbor_nsplit_weight;  // only for intra_skip_thr
  int inherit_skip_reference;            // only used in InitCu
  int inherit_skip_globalmv;             // only used in InitCu
  int inherit_skip_warp;                 // only used in InitCu
  int ref_mask_layer;
  int skip_cu_base_submv[6];
  int try_intra_before_inter;
  int intra_rect_layer_thr;
  int intra_rect4_layer_thr;
  int skip_palette_by_ref;
  int skip_hv4_by_grad[6];
  int skip_4x4_by_stats[7];
  int skip_8x8_by_stats[7];
  int skip_64x64_by_stats[7];
  int skip_128x128_by_stats[7];
  int skip_64_1to4_by_stats[7];
  int skip_32_1to4_by_stats[7];
  int prune_intramode_by_scc;
  int first_stage_split;
  int skip_intra_base_analysis[7];
  int skip_intra_base_best_nb[7];
  int skip_intra_from_comp_mode[6];
  // to rqt
  int prune_tx_search_level;  // TODO: to rqt
  uint8_t tx_prune_mode[6];
  int use_reduced_intra_txset;  // prune two least frequently chosen transforms for each intra mode
  int use_skip_flag_prediction[2];
  int ml_tx_split_thresh;  // Threshold used by the ML based method to predict TX block split decisions.
  int enable_flip_idtx;    // enable flip and identity transform types
  int use_intra_dct_only[2];
  int use_inter_dct_only;
  int luma_txsplit_skip[2];
  int luma_txsplit_inter[6];
  int luma_txsplit_intra[6];
  int inter_txtype_bestref_skip[4];
  int full_rec_skip_full_thresh[6];
  int subcu_skip_from_remain[6];
  int txtype_prune_qsort;
  int txtype_refrd_skip[3];
  int intra_txtype_bestref_skip[7];
  int luma_txdepth_skiptu;
  int fast_rdoq_thres[6];
  int inter_decision_tu_depth[6];
  int luma_intracost_skiptu;
  int calc_tuskip_thresh;
  int full_rd_use_rdoq;
  int fastrdoq_layer;
  int fastrqt_chroma;
  int rdoq_uv_mult[6];
  int rdoq_uv_mult_thr;
  int rdoq_uv_mult_qp;
  int disable_identical_txtype;
  int unit_trans_from_size;
  int enable_ml_tx_split;
  int txfm_type_skip_same_rd[6];
  int intra_txfm_type_yrd[9];
  int fst_inter_intra_txfm_type[8];  // check inter before check intra
  int fst_intra_inter_txfm_type[8];  // check intra before check inter

  // to enc_inter
  int lookahead_mesh_level;
  int encode_mesh_level;  // TODO: search
  int max_mesh_interval;
  int interp_level;
  uint8_t subpel_interp_taps;  // TODO: to enc_inter
  uint8_t subpel_adapt_interp;
  int cdef_skip_layer;
  int restoration_skip_layer;
  int prune_wedge_pred_diff_based;  // TODO: to enc_inter
  int search_method;                // Motion search method (Diamond, NSTEP, Hex, Big Diamond, Square, etc).
  int search_range[2];
  int subpel_iters_per_step[2];               // Maximum number of steps in logarithmic subpel search before giving up.
  uint8_t enable_sec_sub_search;              // enable me second  large mv subpel search
  uint8_t subpel_force_stop;                  // When to stop subpel search.
  uint8_t lookahead_subpel_force_stop;        // When to stop subpel search for tpl and arnr.
  uint8_t joint_search_size_thresh[2];        // TODO: to enc_inter
  uint8_t comp_inter_joint_search_thresh[7];  // TODO: to enc_inter
  int prune_intra_from_skip[7];
  int max_intra_bsize;                // TODO: to enc_inter
  int enable_dist_wtd_comp;           // enable dist wtd compound for sequence
  int enable_reduced_reference_set;   // sequence level
  int enable_masked_comp;             // enable masked compound for sequence
  int enable_onesided_comp;           // enable one sided compound for sequence
  int enable_interintra_comp;         // enable interintra compound for sequence
  int enable_diff_wtd_comp;           // enable diff-wtd compound usage
  int param_enable_interinter_wedge;  // enable interinter-wedge compound usage
  int param_enable_interintra_wedge;  // enable interintra-wedge compound usage
  int enable_mv_refine[6];
  int mv_refine_limit[6];
  int mv_refine_in_cu_layer;
  int enable_mode_threshold;
  int inter_search_order;
  int enable_single_merge;
  int corner_pel_count[2];
  int skip_fullrd_from_skip[6];
  int skip_intra_from_other_partition;
  int mesh_search_thresh;
  int try_inter_from_subintra;
  int skip_intra_dir_num[6];
  int adapt_me_method[3];
  int adapt_me_range[4];
  int start_mv_opt;
  int ref_mv_stack_cand_opt;
  int allow_exhaustive_searches;  // TODO: to enc_inter
  int fast_mesh_search;
  int get_inherit_data;
  int force_get_inherit_data;
  int get_inherit_num;
  int inter_fullrd_skip_same_dist;
  int tz_dia_skip_round;
  int tz_refine;
  int tz_scan_centre_change;
  int intrabc_rd_opt;
  int intrabc_remove_restriction;
  int ref_select_base_single_mode[6];  // 0 for close class,other for slice type
  int ref_use_nb_info;
  uint32_t ref_nblock_thr[6];
  uint32_t ref_simple_thr[6];
  int ref_mask_adjust;
  uint32_t ref_nblock_min;
  int ref_single_thr[6];
  uint32_t ref_newmv_thr[6];
  int ref_newmv_add_base_sad[6];
  int comp_mode_ref_limit[6];
  int comp_mode_ref_limit_estrd[6];
  int enable_fullrd;
  int fullrd_ref_ratio;
  int reduce_newmv_mvp_num[4];
  int handle_inter_newmv[6];
  int handle_inter_new_newmv;
  int fullrd_skip_rd_thr[7];
  int fullrd_pre_break_thr[7];  // before inter_txfm_search for rdo
  int skip_motionmode_after_simple[7];
  int skip_full_rd_thr[6];
  int skip_motionmode_base_refrd[6];
  int skip_comp_mode_thr[6];
  int skip_base_single_merge[6];
  int skip_comp_search_thr[6];
  int first_intra_ratio[6];
  int first_intra_ratio_palette[6];
  uint32_t only_intra_ratio[7];
  int only_inter_ratio[6];
  uint32_t inter_simple_intra[6];
  uint32_t sec_layer_skip_base_var[6];
  uint32_t fst_intra_skip_after_single_mrg[6];
  uint32_t fst_intra_skip_base_remain_rd[6];
  uint32_t fst_intra_skip_comp_search[6];
  uint32_t fst_intra_skip_comp_mode[7];
  uint32_t skip_interp_rd_single;
  uint32_t skip_intra_from_father_skip[6];
  uint32_t skip_interp_motionmode_rd[6];
  uint32_t skip_interp_motionmode_base_mv;
  uint32_t fst_intra_use_tpl;         // 0 use lookahead; other use tpl
  uint32_t fst_intra_use_qpfactor;    // 0 use lookahead; other use tpl
  uint32_t enable_inter_three_layer;  // 0 use two layer; other use three layer
  int qme_early_skip_thr[6];          // Quarter accuracy 1/4
  int eme_early_skip_thr[6];          // One-eighth accuracy  1/8
  int subme_point4_skip_thr[6];       // Subme refine threshold
  int mv_refine_skip_num[6];
  int mv_refine_skip_thr[7];;
  int mv_refine_n1_skip_thr[6];
  int skip_frac_me_encode;
  int skip_frac_me_arnr;
  int skip_frac_me_tpl;
  int adjust_intra_num[6];
  int adjust_intra_thr[6];
  int early_skip_intra_thr[7];
  int early_skip_single_mrg_thr[6];
  int early_skip_single_sr_thr[6];
  int newmv_ref_order_adjust[6];
  int skip_single_sr_ref_order[7];
  int single_merge_rdo[8];
  int reduce_sgl_mrg_rdo_thr[8];
  int me_correct[5];
  int first_rd_ratio;
  int first_rd_push_ratio;
  int second_rd_ratio;
  int second_rd_push_ratio[2];
  int sec_push_rd_thr[6];
  int sec_push_skip_comp_mode[6];
  int enable_10bit_fast_txfm;  // only work for 10bit
  int fullrd_txfm_max_num[6];
  int sgl_mrg_rdo_max_num[6];
  int secrd_txfm_max_num[6];
  int param_fast_obmc_search;  // 0: obmc_full_pixel_diamond 1~6: obmc_refining_search_sad (faster)
  int fast_sgl_mrg_rdo;
  int selective_ref_frame;
  int first_layer_comp_thr;
  int second_layer_comp_thr;
  int inter_intra_thr;
  int reduce_newmv_by_splitnone;
  int newmv_only_first_mvp;
  int first_layer_uv_ref_ratio;
  int skip_obmc_base_partition_bsize[10];  // 0 for enable, 1~6 for bsize, 7 for partition, 8 for mode, 9 for slice_depth
  int skip_warp_base_partition_bsize[10];  // 0 for enable, 1~6 for bsize, 7 for partition, 8 for mode, 9 for slice_depth
  // to enc_intra
  int enable_filter_intra;  // enable filter intra for sequence
  int enable_smooth_intra;  // enable smooth intra modes for sequence
  int enable_paeth_intra;   // enable Paeth intra mode for sequence
  int enable_cfl_intra;     // enable CFL uv intra mode for sequence
  int enable_palette;
  int enable_intrabc;
  int enable_intrabc_hash;
  int intrabc_mesh_range;
  int intrabc_mesh_interval;
  int enable_angle_delta;
  int inter_curvfit_rd_ratio;
  int intra_ydir_decision_tu_depth[6];
  int intra_coarse_precise_size[2];  // 0 intra  1 inter
  int intra_ymode_rd_num[7];         // 0 intra  1 inter
  int ymode_delta_angle_rd_thr[2];   // 0 intra  1 inter
  int intra_ymode_nb_thr[2];         // 0 intra  1 inter
  int intra_ymode_sec_rd_num[7];     // 0 intra  1-6 inter
  int intra_ymode_skip_thr[2];       // 0 intra  1 inter
  int intra_ymode_rd_thr[7];         // 0 intra  1 inter
  int intra_filter_rd_num[6];
  int intra_filter_rd_thr[6];
  int intra_skip_filter_thr[6];
  int intra_uvmode_angle_num[2];  // 0 intra  1 inter
  int intra_uvmode_angle_thr[2];  // 0 intra  1 inter
  int only_try_uv_dm_mode[2];     // 0 intra  1 inter
  int uvmode_skip_angle_thr[2];   // 0 intra  1 inter
  int intra_skip_thr[4];          // only for inter
  int intra_angle_fullrd_skip[7];
  int intra_ymode_angle_skip[7];
  int enable_inter_intra_palette_uv;
  int palette_est_thr[2];
  int palette_skip_param[2];
  int palette_num_reduce;
  uint32_t simple_intra_for_inter[6];
  uint32_t fst_intra_dc_fullrd;
  int intra_pruning_luma_with_hog;
  float intra_prune_luma_with_hog[7];
  float intra_prune_chroma_with_hog[7];
  int intra_pruning_chroma_with_hog;
  int skip_palette_from_minmax;
  int enable_intra_edge_filter;  // enable intra-edge filter for sequence
  int skip_palette_uv_base_y;
  int skip_palette_uv_base_yuv;
  int skip_palette_uv_for_yskip;
  int skip_palette_base_nb[7];
  int intra_ymode_base_toprd[7];  // 0~5 for slice,6 for num
  int skip_cfl_base_nb[2];
  int intra_use_4x4_nb;
  int intra_y_delta_skip_thr[6];
  int intra_y_delta_skip_base_currrd;
  int intra_y_main_skip_base_currrd;
  int intra_y_delta_skip_num[6];
  // to enc_cu to enc_inter
  int get_inherit_mode_ref;
  int enable_interp_switchable;
  int interp_rd_from_luma;
  int close_simple_scene_interp_rd;
  int close_single_interp_rd;
  int close_sharp_interp_rd;
  int sharp_var_thres;
  int prune_inter_8x8;
  int prune_single_merge;
  int prune_comp_merge;
  int prune_single_search;
  int prune_comp_search;
  int prune_comp_sr_mode;
  int skip_newmv_mc;
  int early_terminate_skip_mode;
  int boundary_mode;              // TODO: remove ?
  tenav1_timing_info_t timing_info;  // TODO: remove

  int rest_wiener_start_step;
  int gmv_corner_detect_sample;
  int zero_resi_skip_txfm;
  int wiener_use_window_reduced;

  // AOM RC cfg
  int64_t starting_buffer_level_ms;
  int64_t optimal_buffer_level_ms;
  int64_t maximum_buffer_size_ms;
  unsigned int max_intra_bitrate_pct;
  unsigned int max_inter_bitrate_pct;
  int under_shoot_pct;
  int over_shoot_pct;
  int vbrmin_section;
  int vbrmax_section;
  ///// GFConfig,gf_cfg
  double kf_boost_scale;
  double coded_error_scale;
  double coded_error_scale_low;
  double coded_error_scale_high;
  int boost_weight;

  int adaptive_boost_in_bit_allocation;  // optimize the weight of normal B frames in boost bits calculation
  int sliding_window_scale;              // the scale of sliding window in bits error correction, default is 1
  int adaptive_cqlevel_in_boost;         // use cq level calculated by model and target bits to compute boost in preanalysis
  int align_aom_coded_error;             // whether to align aom coded error or satd
  double avg_satd_scale;                 // scale of avg satd used in active worst quality computation
  double intra_skip_pct_scale;           // align intra skip pcnt
  int use_dc_pct_in_coded_error;
  double adj_r_in_awq_with_boost;   // scale target bits in calculating active worst quality with boost
  int put_forward_tpl_calculation;  // put forward tpl calculation
  int inter_skip_thresh;
  int adjust_worst_quality_with_boost;  // adjust worst quality calculation with boost for GOP of kf
  int add_vbv_aom;
  int use_boost_bre_opt;                 // bite qingxie
  int limit_left_bandwidth;              // limit the adjustment of left bandwidth
  int adj_worst_quality_255;             // adjust intl arf worst quality when GOP worst quality =255
  int adj_frame_worst_quality_overflow;  // adjust frame worst quality when equals to best quality and overflow
  int move_forward_q_update;             // move forward q update
  int open_opt_vbr1190;                  // close above 4 opt
} tenav1_param_t;

typedef void tenav1_t;

void tenav1_param_default(tenav1_param_t *param);
#define TEN_AV1_PARAM_BAD_NAME (-1)
#define TEN_AV1_PARAM_BAD_VALUE (-2)
int tenav1_param_parse(tenav1_param_t *param, const char *name, const char *value);
int tenav1_param_config_preset(tenav1_param_t *param);
tenav1_t *tenav1_encoder_open(tenav1_param_t *para, void ** extra, const char * config);
int32_t tenav1_encoder_encode(tenav1_t *h, const tenav1_inpic_t *inpic,
        tenav1_output_t *outpic, void ** extra);
void tenav1_encoder_close(tenav1_t *h);
const char *tenav1_version(void);

#ifdef __cplusplus
}
#endif
/************************************************************************************************************************************************/

#endif  // SOURCE_LIB_INC_TEN265_H_
